/*
 * Copyright (C) 2021 Authlete, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific
 * language governing permissions and limitations under the
 * License.
 */
package com.authlete.jakarta;


import jakarta.servlet.http.HttpServletRequest;
import jakarta.ws.rs.WebApplicationException;
import jakarta.ws.rs.core.HttpHeaders;
import jakarta.ws.rs.core.Response;
import com.authlete.common.api.AuthleteApi;
import com.authlete.common.dto.GMRequest;
import com.authlete.common.types.GMAction;


/**
 * A base class for grant management endpoint.
 *
 * @since 2.38
 *
 * @see <a href="https://openid.net/specs/fapi-grant-management.html"
 *      >Grant Management for OAuth 2.0</a>
 */
public class BaseGrantManagementEndpoint extends BaseResourceEndpoint
{
    /**
     * Handle a grant management request.
     *
     * <p>
     * This method internally creates a {@link GMRequestHandler} instance and
     * calls its {@link GMRequestHandler#handle(GMRequest) handle}<code>(</code>{@link
     * GMRequest}<code>)</code> method.
     * </p>
     *
     * @param api
     *         An implementation of {@link AuthleteApi}.
     *
     * @param req
     *         An HTTP request that the grant management endpoint received.
     *         Its HTTP method must be either {@code GET} or {@code DELETE}.
     *
     * @param grantId
     *         The grant ID included in the API call to the grant management
     *         endpoint. The last part of the path of the URL.
     *
     * @return
     *         A response that should be returned to the client application.
     *
     * @throws WebApplicationException
     *         An exception generated by {@link GMRequestHandler}.
     */
    public Response handle(
            AuthleteApi api, HttpServletRequest req, String grantId)
                    throws WebApplicationException
    {
        // Construct request parameters for an API call to the /api/gm API.
        GMRequest request = buildRequest(req, grantId);

        // Create a handler that communicates with the /api/gm API.
        GMRequestHandler handler = new GMRequestHandler(api);

        // Call the /api/gm API and handle the response.
        return handler.handle(request);
    }


    /**
     * Construct request parameters for an API call to the /api/gm API.
     */
    private GMRequest buildRequest(HttpServletRequest req, String grantId)
    {
        return new GMRequest()
                .setGmAction(determineGmAction(req))
                .setGrantId(grantId)
                .setAccessToken(extractAccessToken(req))
                .setClientCertificate(extractClientCertificate(req))
                .setDpop(extractDpop(req))
                ;
    }


    /**
     * Determine the value of the gmAction parameter from the HTTP method.
     */
    private static GMAction determineGmAction(HttpServletRequest req)
    {
        if ("DELETE".equalsIgnoreCase(req.getMethod()))
        {
            return GMAction.REVOKE;
        }
        else
        {
            return GMAction.QUERY;
        }
    }


    /**
     * Extract an access token from the Authorization header.
     */
    private String extractAccessToken(HttpServletRequest req)
    {
        return extractAccessToken(
                req.getHeader(HttpHeaders.AUTHORIZATION), null);
    }


    /**
     * Extract the value of the DPoP header.
     */
    private String extractDpop(HttpServletRequest req)
    {
        return req.getHeader("DPoP");
    }
}
