/*
 * Decompiled with CFR 0.152.
 */
package com.azure.security.keyvault.jca.implementation;

import com.azure.security.keyvault.jca.implementation.KeyVaultPrivateKey;
import com.azure.security.keyvault.jca.implementation.model.AccessToken;
import com.azure.security.keyvault.jca.implementation.model.CertificateBundle;
import com.azure.security.keyvault.jca.implementation.model.CertificateItem;
import com.azure.security.keyvault.jca.implementation.model.CertificateListResult;
import com.azure.security.keyvault.jca.implementation.model.CertificatePolicy;
import com.azure.security.keyvault.jca.implementation.model.KeyProperties;
import com.azure.security.keyvault.jca.implementation.model.SecretBundle;
import com.azure.security.keyvault.jca.implementation.model.SignResult;
import com.azure.security.keyvault.jca.implementation.utils.AccessTokenUtil;
import com.azure.security.keyvault.jca.implementation.utils.HttpUtil;
import com.azure.security.keyvault.jca.implementation.utils.JsonConverterUtil;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.StringReader;
import java.net.URLEncoder;
import java.security.Key;
import java.security.KeyFactory;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.ArrayList;
import java.util.Base64;
import java.util.HashMap;
import java.util.List;
import java.util.Optional;
import java.util.logging.Level;
import java.util.logging.Logger;

public class KeyVaultClient {
    private static final Logger LOGGER = Logger.getLogger(KeyVaultClient.class.getName());
    private final String keyVaultBaseUri;
    private final String keyVaultUri;
    private final String tenantId;
    private final String clientId;
    private final String clientSecret;
    private String managedIdentity;
    private AccessToken accessToken;
    private final boolean disableChallengeResourceVerification;

    KeyVaultClient(String keyVaultUri, String managedIdentity) {
        this(keyVaultUri, null, null, null, managedIdentity, false);
    }

    public KeyVaultClient(String keyVaultUri, String tenantId, String clientId, String clientSecret) {
        this(keyVaultUri, tenantId, clientId, clientSecret, null, false);
    }

    public KeyVaultClient(String keyVaultUri, String tenantId, String clientId, String clientSecret, String managedIdentity, boolean disableChallengeResourceVerification) {
        LOGGER.log(Level.INFO, "Using Azure Key Vault: {0}", keyVaultUri);
        this.keyVaultUri = HttpUtil.addTrailingSlashIfRequired(HttpUtil.validateUri(keyVaultUri, "Azure Key Vault URI"));
        String domainNameSuffix = Optional.of(this.keyVaultUri).map(uri -> uri.split("\\.", 2)[1]).map(suffix -> suffix.substring(0, suffix.length() - 1)).orElse(null);
        this.keyVaultBaseUri = "https://" + domainNameSuffix;
        this.tenantId = tenantId;
        this.clientId = clientId;
        this.clientSecret = clientSecret;
        this.managedIdentity = managedIdentity;
        this.disableChallengeResourceVerification = disableChallengeResourceVerification;
    }

    public static KeyVaultClient createKeyVaultClientBySystemProperty() {
        String keyVaultUri = System.getProperty("azure.keyvault.uri");
        String tenantId = System.getProperty("azure.keyvault.tenant-id");
        String clientId = System.getProperty("azure.keyvault.client-id");
        String clientSecret = System.getProperty("azure.keyvault.client-secret");
        String managedIdentity = System.getProperty("azure.keyvault.managed-identity");
        boolean disableChallengeResourceVerification = Boolean.parseBoolean(System.getProperty("azure.keyvault.disable-challenge-resource-verification"));
        return new KeyVaultClient(keyVaultUri, tenantId, clientId, clientSecret, managedIdentity, disableChallengeResourceVerification);
    }

    private String getAccessToken() {
        if (this.accessToken != null && !this.accessToken.isExpired()) {
            return this.accessToken.getAccessToken();
        }
        this.accessToken = this.getAccessTokenByHttpRequest();
        return this.accessToken.getAccessToken();
    }

    private AccessToken getAccessTokenByHttpRequest() {
        LOGGER.entering("KeyVaultClient", "getAccessTokenByHttpRequest");
        AccessToken accessToken = null;
        try {
            String resource = URLEncoder.encode(this.keyVaultBaseUri, "UTF-8");
            if (this.managedIdentity != null) {
                this.managedIdentity = URLEncoder.encode(this.managedIdentity, "UTF-8");
            }
            if (this.tenantId != null && this.clientId != null && this.clientSecret != null) {
                String aadAuthenticationUri = AccessTokenUtil.getLoginUri(this.keyVaultUri + "certificates" + "?api-version=7.1", this.disableChallengeResourceVerification);
                accessToken = AccessTokenUtil.getAccessToken(resource, aadAuthenticationUri, this.tenantId, this.clientId, this.clientSecret);
            } else {
                accessToken = AccessTokenUtil.getAccessToken(resource, this.managedIdentity);
            }
        }
        catch (Throwable t) {
            LOGGER.log(Level.WARNING, "Could not obtain access token to authenticate with.", t);
        }
        LOGGER.exiting("KeyVaultClient", "getAccessTokenByHttpRequest", accessToken);
        return accessToken;
    }

    public List<String> getAliases() {
        ArrayList<String> result = new ArrayList<String>();
        HashMap<String, String> headers = new HashMap<String, String>();
        headers.put("Authorization", "Bearer " + this.getAccessToken());
        String uri = this.keyVaultUri + "certificates" + "?api-version=7.1";
        while (uri != null && !uri.isEmpty()) {
            String response = HttpUtil.get(uri, headers);
            CertificateListResult certificateListResult = null;
            if (response != null) {
                certificateListResult = (CertificateListResult)JsonConverterUtil.fromJson(response, CertificateListResult.class);
            }
            if (certificateListResult != null) {
                uri = certificateListResult.getNextLink();
                for (CertificateItem certificateItem : certificateListResult.getValue()) {
                    String id = certificateItem.getId();
                    String alias = id.substring(id.indexOf("certificates") + "certificates".length() + 1);
                    result.add(alias);
                }
                continue;
            }
            uri = null;
        }
        return result;
    }

    private CertificateBundle getCertificateBundle(String alias) {
        CertificateBundle result = null;
        HashMap<String, String> headers = new HashMap<String, String>();
        headers.put("Authorization", "Bearer " + this.getAccessToken());
        String uri = this.keyVaultUri + "certificates/" + alias + "?api-version=7.1";
        String response = HttpUtil.get(uri, headers);
        if (response != null) {
            result = (CertificateBundle)JsonConverterUtil.fromJson(response, CertificateBundle.class);
        }
        return result;
    }

    public Certificate getCertificate(String alias) {
        String certificateString;
        LOGGER.entering("KeyVaultClient", "getCertificate", alias);
        LOGGER.log(Level.INFO, "Getting certificate for alias: {0}", alias);
        X509Certificate certificate = null;
        CertificateBundle certificateBundle = this.getCertificateBundle(alias);
        if (certificateBundle != null && (certificateString = certificateBundle.getCer()) != null) {
            try {
                CertificateFactory cf = CertificateFactory.getInstance("X.509");
                certificate = (X509Certificate)cf.generateCertificate(new ByteArrayInputStream(Base64.getDecoder().decode(certificateString)));
            }
            catch (CertificateException ce) {
                LOGGER.log(Level.WARNING, "Certificate error", ce);
            }
        }
        LOGGER.exiting("KeyVaultClient", "getCertificate", certificate);
        return certificate;
    }

    public Key getKey(String alias, char[] password) {
        LOGGER.entering("KeyVaultClient", "getKey", new Object[]{alias, password});
        LOGGER.log(Level.INFO, "Getting key for alias: {0}", alias);
        CertificateBundle certificateBundle = this.getCertificateBundle(alias);
        boolean isExportable = Optional.ofNullable(certificateBundle).map(CertificateBundle::getPolicy).map(CertificatePolicy::getKeyProperties).map(KeyProperties::isExportable).orElse(false);
        String keyType = Optional.ofNullable(certificateBundle).map(CertificateBundle::getPolicy).map(CertificatePolicy::getKeyProperties).map(KeyProperties::getKty).orElse(null);
        if (!isExportable) {
            LOGGER.exiting("KeyVaultClient", "getKey", null);
            String keyType2 = keyType.contains("-HSM") ? keyType.substring(0, keyType.indexOf("-HSM")) : keyType;
            return Optional.ofNullable(certificateBundle).map(CertificateBundle::getKid).map(kid -> new KeyVaultPrivateKey(keyType2, (String)kid, this)).orElse(null);
        }
        String certificateSecretUri = certificateBundle.getSid();
        HashMap<String, String> headers = new HashMap<String, String>();
        headers.put("Authorization", "Bearer " + this.getAccessToken());
        String body = HttpUtil.get(certificateSecretUri + "?api-version=7.1", headers);
        if (body == null) {
            LOGGER.exiting("KeyVaultClient", "getKey", null);
            return null;
        }
        Key key = null;
        SecretBundle secretBundle = (SecretBundle)JsonConverterUtil.fromJson(body, SecretBundle.class);
        String contentType = secretBundle.getContentType();
        if ("application/x-pkcs12".equals(contentType)) {
            try {
                KeyStore keyStore = KeyStore.getInstance("PKCS12");
                keyStore.load(new ByteArrayInputStream(Base64.getDecoder().decode(secretBundle.getValue())), "".toCharArray());
                alias = keyStore.aliases().nextElement();
                key = keyStore.getKey(alias, "".toCharArray());
            }
            catch (IOException | KeyStoreException | NoSuchAlgorithmException | UnrecoverableKeyException | CertificateException e) {
                LOGGER.log(Level.WARNING, "Unable to decode key", e);
            }
        } else if ("application/x-pem-file".equals(contentType)) {
            try {
                key = this.createPrivateKeyFromPem(secretBundle.getValue(), keyType);
            }
            catch (IOException | IllegalArgumentException | NoSuchAlgorithmException | InvalidKeySpecException ex) {
                LOGGER.log(Level.WARNING, "Unable to decode key", ex);
            }
        }
        LOGGER.exiting("KeyVaultClient", "getKey", key);
        return key;
    }

    public byte[] getSignedWithPrivateKey(String digestName, String digestValue, String keyId) {
        SignResult result = null;
        String bodyString = String.format("{\"alg\": \"" + digestName + "\", \"value\": \"%s\"}", digestValue);
        HashMap<String, String> headers = new HashMap<String, String>();
        headers.put("Authorization", "Bearer " + this.getAccessToken());
        String uri = keyId + "/sign" + "?api-version=7.1";
        String response = HttpUtil.post(uri, headers, bodyString, "application/json");
        if (response != null) {
            result = (SignResult)JsonConverterUtil.fromJson(response, SignResult.class);
        }
        if (result != null) {
            return Base64.getUrlDecoder().decode(result.getValue());
        }
        return new byte[0];
    }

    private PrivateKey createPrivateKeyFromPem(String pemString, String keyType) throws IOException, NoSuchAlgorithmException, InvalidKeySpecException {
        StringBuilder builder = new StringBuilder();
        try (BufferedReader reader = new BufferedReader(new StringReader(pemString));){
            String line = reader.readLine();
            if (line == null || !line.contains("BEGIN PRIVATE KEY")) {
                throw new IllegalArgumentException("No PRIVATE KEY found");
            }
            line = "";
            while (line != null) {
                if (line.contains("END PRIVATE KEY")) {
                    break;
                }
                builder.append(line);
                line = reader.readLine();
            }
        }
        byte[] bytes = Base64.getDecoder().decode(builder.toString());
        PKCS8EncodedKeySpec spec = new PKCS8EncodedKeySpec(bytes);
        KeyFactory factory = KeyFactory.getInstance(keyType);
        return factory.generatePrivate(spec);
    }
}

