/*
 * Copyright 2014 Blazebit.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.blazebit.persistence.impl;

import com.blazebit.persistence.impl.expression.FunctionExpression;
import com.blazebit.persistence.impl.expression.PathExpression;
import com.blazebit.persistence.impl.expression.SubqueryExpression;
import com.blazebit.persistence.impl.expression.VisitorAdapter;

/**
 *
 * @author Moritz Becker
 * @since 1.0
 */
public class JoinVisitor extends VisitorAdapter {

    private final JoinManager joinManager;
    private boolean joinWithObjectLeafAllowed = true;
    private ClauseType fromClause;
    

    public JoinVisitor(JoinManager joinManager) {
        this.joinManager = joinManager;
    }

    public ClauseType getFromClause() {
        return fromClause;
    }

    public void setFromClause(ClauseType fromClause) {
        this.fromClause = fromClause;
    }

    @Override
    public void visit(PathExpression expression) {
        joinManager.implicitJoin(expression, joinWithObjectLeafAllowed, fromClause, false, false);
    }

    @Override
    public void visit(FunctionExpression expression) {
        // do not join outer expressions
        if(!ExpressionUtils.isOuterFunction(expression)){
            super.visit(expression);
        }
    }
    
    
    
    // Added eager initialization of subqueries
    @Override
    public void visit(SubqueryExpression expression) {
        // TODO: we have to pass the fromClause into the subquery so that joins, generated by OUTERs from the subquery don't get rendered if not needed
        ((SubqueryBuilderImpl<?>) expression.getSubquery()).applyImplicitJoins();
    }

    public boolean isJoinWithObjectLeafAllowed() {
        return joinWithObjectLeafAllowed;
    }

    public void setJoinWithObjectLeafAllowed(boolean joinWithObjectLeafAllowed) {
        this.joinWithObjectLeafAllowed = joinWithObjectLeafAllowed;
    }
}
