package com.clover.sdk.internal.util.calc;

import com.clover.sdk.internal.util.Lists;
import com.clover.sdk.internal.util.Maps;

import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * <p>Generalized order calculation logic.  Calculates all amounts associated with an order, including
 * total, subtotal, discounts, tips, service charges, and tax.</p>
 * <p/>
 * Some conventions that implementers must adhere to, to keep things simple and consistent:
 * <ul>
 * <li>All amounts (including discounts) are positive.</li>
 * <li>For refunds, {@link com.clover.sdk.internal.util.calc.Calc.LineItem#getPrice()} can return a negative number as long as
 * {@link com.clover.sdk.internal.util.calc.Calc.LineItem#allowNegativePrice()} returns {@code true}.</li>
 * <li>Methods that return {@link java.util.Collection} can never return {@code null} unless specified.</li>
 * <li>Percent and tax rate values are a number between {@code 0} and {@code 100}.</li>
 * </ul>
 */
public class Calc {

  public static final Decimal HUNDRED = new Decimal(100);

  /**
   * Using a fixed scale on numbers whose
   */
  public static Decimal divideFixedScale(Decimal dividend, Decimal divisor) {
    return dividend.divide(divisor, 7, RoundingMode.HALF_UP);
  }

  private final Order order;
  private final Validator validator;
  private final Logger log;

  public Calc(Order order) {
    this(order, null);
  }

  public Calc(Order order, Logger log) {
    if (order == null) {
      throw new NullPointerException("order cannot be null");
    }
    if (log == null) {
      log = new NoLog();
    }
    this.order = order;
    this.log = log;
    validator = new Validator(order, log);
  }

  public interface Order {
    public boolean isVat();

    public boolean isTaxRemoved();

    public Collection<? extends LineItem> getLineItems();

    public Price getComboDiscount();

    public Price getAmountDiscount();

    public Decimal getPercentDiscount();

    public Price getTip();

    public Decimal getPercentServiceCharge();
  }

  public interface LineItem {
    public Price getPrice();

    public boolean allowNegativePrice();

    public Decimal getUnitQuantity();

    public boolean isRefunded();

    public boolean isExchanged();

    public Collection<Decimal> getTaxRates();

    public Price getModification();

    public Price getAmountDiscount();

    public Decimal getPercentDiscount();

    public Decimal getSplitPercent();
  }

  public interface Logger {
    public void warn(String s);
  }

  private class NoLog implements Logger {
    @Override
    public void warn(String s) {
    }
  }

  public static class TaxSummary {
    public final Price gross;
    public final Price net;
    public final Price tax;
    public final Decimal taxRate;

    public TaxSummary(Price gross, Price net, Price tax, Decimal taxRate) {
      this.gross = gross;
      this.net = net;
      this.tax = tax;
      this.taxRate = taxRate;
    }
  }

  public static class PaymentDetails {
    public final List<TaxSummary> taxSummaries;
    public final Price serviceCharge;

    private PaymentDetails(List<TaxSummary> taxSummaries, Price serviceCharge) {
      this.taxSummaries = Collections.unmodifiableList(taxSummaries);
      this.serviceCharge = serviceCharge;
    }
  }

  /**
   * Return the line item price, multiplied by the unit quantity.  The result is rounded to
   * 2 decimal places, to avoid rounding errors when adding lines together.
   */
  public Price getExtendedPrice(LineItem line) {
    Price price = validator.getPrice(line);
    Decimal unitQty = validator.getUnitQuantity(line);
    Decimal splitRatio = divideFixedScale(validator.getSplitPercent(line), HUNDRED);
    return price.multiply(unitQty).multiply(splitRatio).round();
  }

  /**
   * Return the extended price with modifications and discounts applied.  The result is rounded to
   * 2 decimal places, to avoid rounding errors when adding lines together.
   */
  public Price getPriceWithModificationsAndDiscounts(LineItem line) {
    // Calculate line item price.
    Price originalPrice = getExtendedPrice(line);

    // Nasty hack for refunds.  If the line item price is negative, assume that this is a manual
    // refund.  We ignore modifications and discounts because they don't make sense with negative
    // prices.  We get the line item price from Validator, which ensures that a negative price is
    // only returned when allowed.
    if (originalPrice.isLessThan(Price.ZERO)) {
      return originalPrice;
    }

    // If this item is being split, adjust the modification and amount discount.
    Decimal splitRatio = divideFixedScale(validator.getSplitPercent(line), HUNDRED);

    // Add modification.
    Price modification = validator.getModification(line).multiply(splitRatio).round();
    originalPrice = originalPrice.add(modification);

    // Subtract discounts.
    Decimal percentDiscount = validator.getLinePercentDiscount(line);
    Price percentDiscountAmount = originalPrice.multiply(divideFixedScale(percentDiscount, HUNDRED)).round();
    Price amountDiscount = validator.getLineAmountDiscount(line).multiply(splitRatio).round();
    Price totalDiscount = percentDiscountAmount.add(amountDiscount);
    if (totalDiscount.isGreaterThan(originalPrice)) {
      // Don't allow discount to be greater than the item price.
      return Price.ZERO;
    }

    return originalPrice.subtract(totalDiscount);
  }

  /**
   * Return the line subtotal with combo discounts applied.  Order-level discounts are not applied.
   * Combo discounts are applied.
   *
   * @see #getLineSubtotal(java.util.Collection)
   * @see //com.clover.core.internal.calc.Calc.Order#getComboDiscount()
   */
  public Price getLineSubtotal() {
    Price lineSubtotal = getLineSubtotal(order.getLineItems());
    return lineSubtotal.subtract(validator.getComboDiscount());
  }

  /**
   * Return the sum of line items with modifiers, line-level discounts applied.  Line items
   * that are refunded or exchanged are not included.  Order-level discounts are not applied.
   * Combo discounts are not applied, since we don't know if the given lines are the ones that
   * the combo discount applies to.
   */
  public Price getLineSubtotal(Collection<? extends LineItem> lineItems) {
    if (lineItems == null) {
      return Price.ZERO;
    }

    // Add up line items that have not been exchanged or refunded.
    Price subtotal = Price.ZERO;
    for (LineItem line : lineItems) {
      if (!line.isExchanged() && !line.isRefunded()) {
        subtotal = subtotal.add(getPriceWithModificationsAndDiscounts(line));
      }
    }
    return subtotal;
  }

  /**
   * Return the discount multiplier for the given order.  This is the discounted subtotal divided
   * by the line subtotal (before order discounts).  Any component of the order can be multiplied
   * by this number to calculate the post-discount amount.  This value is {@code 1} if no
   * order-level discounts exist.
   *
   * @see #getDiscountedSubtotal
   * @see #getLineSubtotal
   */
  public Decimal getDiscountMultiplier() {
    Price subtotal = getLineSubtotal(order.getLineItems());
    if (subtotal.compareTo(Decimal.ZERO) <= 0) {
      // Don't divide by zero (CLOVER-1702).
      // Subtotal is negative for manual refunds (CLOVER-1985).  See getPriceWithModificationsAndDiscounts().
      return Decimal.ONE;
    }

    Decimal percentDiscount = validator.getOrderPercentDiscount();
    Price percentDiscountAmount = subtotal.multiply(percentDiscount).divide(HUNDRED).round();
    Price discount = percentDiscountAmount.add(validator.getOrderAmountDiscount()).add(validator.getComboDiscount());
    if (discount.isGreaterThan(subtotal)) {
      return Decimal.ZERO;
    }
    Price discountedSubtotal = subtotal.subtract(discount);
    return divideFixedScale(discountedSubtotal, subtotal);
  }

  /**
   * Return the subtotal of all line items with order-level discounts applied.
   */
  public Price getDiscountedSubtotal() {
    return getDiscountedSubtotal(order.getLineItems());
  }

  /**
   * Return the subtotal of the given line items with order-level discounts and combo discounts applied.
   */
  public Price getDiscountedSubtotal(Collection<? extends LineItem> lineItems) {
    Price lineSubtotal = getLineSubtotal(lineItems);
    return lineSubtotal.multiply(getDiscountMultiplier()).round();
  }

  /**
   * Return a {@code List} of tax summaries that specify the gross, net, and tax for each tax rate specified
   * for all the line items in the given order.
   */
  public List<TaxSummary> getTaxSummaries() {
    return getTaxSummaries(order.getLineItems());
  }

  /**
   * Return a {@code List} of tax summaries that specify the gross, net, and tax for each tax rate specified
   * for the given line items.
   */
  public List<TaxSummary> getTaxSummaries(Collection<? extends LineItem> lineItems) {
    List<TaxSummary> summaries = Lists.newArrayList();
    if (order.isTaxRemoved()) {
      return summaries;
    }

    Map<Decimal, List<LineItem>> lineItemsByTaxRate = groupByTaxRate(lineItems);
    for (Map.Entry<Decimal, List<LineItem>> entry : lineItemsByTaxRate.entrySet()) {
      Decimal taxRate = entry.getKey();
      List<LineItem> linesForThisRate = entry.getValue();
      if (order.isVat()) {
        Price gross = getDiscountedSubtotal(linesForThisRate);
        // If the tax rate is 8.75, net = gross / 1.0875.
        Decimal taxDivisor = divideFixedScale(taxRate, HUNDRED).add(Decimal.ONE);
        Price net = gross.divide(taxDivisor).round();
        Price tax = gross.subtract(net);
        summaries.add(new TaxSummary(gross, net, tax, taxRate));
      } else {
        Price net = getDiscountedSubtotal(linesForThisRate);
        Price tax = net.multiply(taxRate).divide(HUNDRED).round();
        Price gross = net.add(tax);
        summaries.add(new TaxSummary(gross, net, tax, taxRate));
      }
    }
    return summaries;
  }

  static Map<Decimal, List<LineItem>> groupByTaxRate(Collection<? extends LineItem> lineItems) {
    Map<Decimal, List<LineItem>> map = Maps.newHashMap();
    if (lineItems == null) {
      return map;
    }
    for (LineItem line : lineItems) {
      List<Decimal> taxRates = new ArrayList<Decimal>(line.getTaxRates());
      if (taxRates.isEmpty()) {
        // Implicitly add a zero-tax summary, so that the caller has the gross and net totals of
        // items that were not taxed.
        taxRates.add(Decimal.ZERO);
      }

      for (Decimal taxRate : taxRates) {
        List<LineItem> linesForTaxRate = map.get(taxRate);
        if (linesForTaxRate == null) {
          linesForTaxRate = Lists.newArrayList();
          map.put(taxRate, linesForTaxRate);
        }
        linesForTaxRate.add(line);
      }
    }
    return map;
  }

  public Price getServiceCharge() {
    return getServiceCharge(order.getLineItems());
  }

  public Price getServiceCharge(Collection<? extends LineItem> lines) {
    Decimal percentServiceCharge = validator.getPercentServiceCharge();
    Price subtotal = getDiscountedSubtotal(lines);
    return subtotal.multiply(percentServiceCharge).divide(HUNDRED).round();
  }

  public Price getTax() {
    return getTax(order.getLineItems());
  }

  public Price getTax(Collection<? extends LineItem> lines) {
    Price tax = Price.ZERO;
    for (TaxSummary ts : getTaxSummaries(lines)) {
      tax = tax.add(ts.tax);
    }
    return tax;
  }

  /**
   * Returns the order total, which includes the discounted subtotal, service charge, and tax.
   * Note that tip is not included in the total.
   *
   * @see #getDiscountedSubtotal
   * @see #getServiceCharge
   * @see #getTax
   */
  public Price getTotal() {
    return getTotal(order.getLineItems());
  }

  /**
   * Returns the order total for the specified line items, which includes the discounted subtotal,
   * service charge, and tax.  Note that tip is not included in the total.
   *
   * @see #getDiscountedSubtotal
   * @see #getServiceCharge
   * @see #getTax
   */
  public Price getTotal(Collection<? extends LineItem> lines) {
    // Note that this code calculates the subtotal multiple times.  We can
    // optimize later if it results in performance problems.
    Price subtotal = getDiscountedSubtotal(lines);
    Price tax = Price.ZERO;
    if (!order.isVat()) {
      tax = getTax(lines);
    }
    Price serviceCharge = getServiceCharge(lines);
    return subtotal.add(tax).add(serviceCharge);
  }


  public PaymentDetails getPaymentDetails(Price payment) {
    return getPaymentDetails(payment, order.getLineItems());
  }

  /**
   * Return the payment breakdown, based on a payment amount which can be either full or partial.
   */
  public PaymentDetails getPaymentDetails(Price payment, Collection<? extends LineItem> lines) {
    if (payment.isLessThan(Price.ZERO)) {
      log.warn("Attempted to get payment details for negative amount: " + payment + ".  Using 0 instead.");
      payment = Price.ZERO;
    }
    // Add up tax summaries and service charge.
    List<TaxSummary> taxSummaries = getTaxSummaries(lines);
    Price serviceCharge = getServiceCharge(lines);
    Price total = getTotal(lines);
    if (payment.isGreaterThan(total)) {
      log.warn("Attempted to overpay.  Calculating payment details based on a payment of " +
          total + " instead of " + payment);
      payment = total;
    }

    // Calculate final tax summaries and service charge based on the portion of the total paid.
    Decimal paymentPercent;
    if (total.equals(Price.ZERO)) {
      // Avoid division by zero.
      paymentPercent = Decimal.ONE;
    } else {
      paymentPercent = divideFixedScale(payment, total);
    }

    List<TaxSummary> partialTaxSummaries = new ArrayList<TaxSummary>();
    for (TaxSummary ts : taxSummaries) {
      Price gross = ts.gross.multiply(paymentPercent).round();
      Price tax = ts.tax.multiply(paymentPercent).round();
      Price net = gross.subtract(tax);
      TaxSummary partialTaxSummary = new TaxSummary(gross, net, tax, ts.taxRate);
      partialTaxSummaries.add(partialTaxSummary);
    }

    return new PaymentDetails(partialTaxSummaries, serviceCharge.multiply(paymentPercent).round());
  }
}
