package com.clover.sdk.internal.util.calc;

import android.os.Parcel;
import android.os.Parcelable;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.DecimalFormat;

/**
 * Immutable signed decimal numbers with a limited range and precision. The goal of the Decimal class is to provide the
 * same basic functionality as BigDecimal except for the unlimited range and precision and with much higher performance.
 * Overflow and NaN behavior is undefined.
 */
public class Decimal extends Number implements Comparable<Decimal>, Parcelable {

  public static final int DEFAULT_SCALE = 7;
  public static final int MAX_SCALE = 9;
  static final int[] MULTIPLIERS = new int[MAX_SCALE+1];

  public static final double MIN_FRACTION;

  static {
    for (int i = 0; i < MULTIPLIERS.length; i++) {
      MULTIPLIERS[i] = pow(10, i);
    }

    MIN_FRACTION = 1 / MULTIPLIERS[MAX_SCALE];
  }

  public static final Decimal ZERO = new Decimal(0, 0);
  public static final Decimal ONE = new Decimal(1, 0);

  protected final double value;
  protected final byte scale;

  /**
   * Construct a new Decimal with the specified value and scale.
   */
  public Decimal(double dbl, int scale) {
    value = roundDouble(dbl, MULTIPLIERS[scale]);
    this.scale = (byte) scale;
  }

  /**
   * Construct a new Decimal with the DEFAULT_SCALE. Prefer explicitly setting scale with {@link #Decimal(double, int)}.
   * @deprecated
   */
  public Decimal(double dbl) {
    this(dbl, DEFAULT_SCALE);
  }

  /**
   * Construct a new Decimal that is a copy of the given Decimal.
   */
  public Decimal(Decimal dval) {
    value = dval.value;
    scale = dval.scale;
  }

  /**
   * Construct a new Decimal from the given Decimal with the same value but a different scale.
   */
  public Decimal(Decimal dval, int newScale) {
    if (newScale < dval.scale) {
      value = roundDouble(dval.value, MULTIPLIERS[newScale]);
    } else {
      value = dval.value;
    }
    scale = (byte) newScale;
  }

  /**
   * Construct a new Decimal with the given long using the DEFAULT scale. It does not divide the long like the
   * {@link #Decimal(long, int)} constructor!
   * @deprecated
   */
  public Decimal(long lval) {
    this((double) lval, DEFAULT_SCALE);
  }

  /**
   * Construct a new Decimal with the value set to (lval / 10^scale) and the specified scale.
   */
  public Decimal(long lval, int scale) {
    this(((double) lval) / MULTIPLIERS[scale], scale);
  }

  /**
   * Construct a new Decimal from the given string. The scale is determined by the number of places after the decimal
   * point. The following regex describes acceptable input: [+-]?(([0-9]*\.[0-9]+)|[0-9]+). This function will not
   * accept a more precise value than can be stored accurately.
   */
  public Decimal(String str) {
    final int start;
    final boolean negative;

    final char firstChar = str.charAt(0);
    if (firstChar == '-') {
      negative = true;
      start = 1;
    } else if (firstChar == '+') {
      negative = false;
      start = 1;
    } else {
      negative = false;
      start = 0;
    }

    long res = 0;
    int precision = 0;

    for (int i = start, length = str.length(); i < length; i++) {
      final char c = str.charAt(i);
      if (c >= '0' && c <= '9') {
        res = res * 10 + (c - '0');
      } else if (c == '.') {
        if (precision > 0) {
          throw new IllegalArgumentException("Input String contains too many decimal points: " + str);
        }
        precision = length - i - 1;
      } else {
        throw new IllegalArgumentException();
      }
    }

    if (precision > MAX_SCALE) {
      throw new IllegalArgumentException("Input String has more precision than can be represented accurately: " + str);
    }

    double v = (double)res / MULTIPLIERS[precision];
    value = negative ? -v : v;
    scale = (byte)precision;
  }

  public Decimal add(Decimal dval) {
    Decimal result = new Decimal(value + dval.value, Math.max(scale, dval.scale));
    return result;
  }

  public Decimal subtract(Decimal dval) {
    Decimal result = new Decimal(value - dval.value, Math.max(scale, dval.scale));
    return result;
  }

  public Decimal divide(Decimal dval) {
    Decimal result = new Decimal(value / dval.value, Math.max(scale, dval.scale));
    return result;
  }

  public Decimal divide(long lval) {
    Decimal result = new Decimal(value / (double) lval, scale);
    return result;
  }

  public Decimal divide(Decimal dval, int newScale, RoundingMode newRoundingMode) {
    if (newRoundingMode != RoundingMode.HALF_UP) {
      throw new IllegalArgumentException();
    }
    return new Decimal(value / dval.value, newScale);
  }

  public Decimal multiply(Decimal dval) {
    Decimal result = new Decimal(value * dval.value, Math.max(scale, dval.scale));
    return result;
  }

  public Decimal multiply(BigDecimal bdval) {
    Decimal result = new Decimal(value * bdval.doubleValue(), Math.max(scale, bdval.scale()));
    return result;
  }

  public Decimal multiply(long lval) {
    Decimal result = new Decimal(value * (double) lval, scale);
    return result;
  }

  public Decimal multiply(Decimal dval, int newScale, RoundingMode newRoundingMode) {
    if (newRoundingMode != RoundingMode.HALF_UP) {
      throw new IllegalArgumentException();
    }
    Decimal result = new Decimal(value * dval.value, newScale);
    return result;
  }

  public Decimal setScale(int newScale, RoundingMode newRoundingMode) {
    if (newRoundingMode != RoundingMode.HALF_UP) {
      throw new IllegalArgumentException("RoundingMode.HALF_UP is the only accepted rounding mode");
    }

    if (scale == newScale) {
      return this;
    }

    return new Decimal(this, newScale);
  }

  @Override
  public double doubleValue() {
    return value;
  }

  @Override
  public float floatValue() {
    return (float) value;
  }

  @Override
  public long longValue() {
    return (long) value;
  }

  @Override
  public int intValue() {
    return (int) value;
  }

  protected static double roundDouble(double value, int multiplier) {
    return Math.floor((value * multiplier) + 0.5d) / multiplier;
  }

  @Override
  public boolean equals(Object o) {
    if (o == this) {
      return true;
    }
    if (o != null && o instanceof Decimal) {
      Decimal d = (Decimal) o;
      return value == d.value;
    }
    return false;
  }

  @Override
  public int hashCode() {
    long bits = Double.doubleToLongBits(value);
    return (int) (bits ^ (bits >>> 32));
  }

  @Override
  public int compareTo(Decimal o) {
    if (equals(o)) {
      return 0;
    } else if (value > o.value) {
      return 1;
    } else {
      return -1;
    }
  }

  @Override
  public String toString() {
    return fasterFormat(value, scale);
  }

  /**
   * Construct a new Decimal by first converting the double to a String with the maximum scale allowed and then parsing
   * that String.
   */
  public static Decimal valueOf(double dval) {
    DecimalFormat df = new DecimalFormat("#");
    df.setMaximumFractionDigits(MAX_SCALE);
    return new Decimal(df.format(dval));
  }

  protected static String fasterFormat(double val, int precision) {
    StringBuilder sb = new StringBuilder();
    if (val < 0) {
      sb.append('-');
      val = -val;
    }
    final int exp = MULTIPLIERS[precision];
    final long lval = (long)(val * exp + 0.5);
    sb.append(lval / exp);
    if (precision > 0) {
      sb.append('.');
      final long fval = lval % exp;
      for (int p = precision - 1; p > 0 && fval < MULTIPLIERS[p]; p--) {
        sb.append('0');
      }
      sb.append(fval);
    }
    return sb.toString();
  }

  protected static int pow(int a, int b) {
    if (b == 0) {
      return 1;
    }
    int result = a;
    while (--b > 0) {
      result *= a;
    }
    return result;
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel out, int flags) {
    out.writeDouble(value);
    out.writeByte(scale);
  }

  public static final Parcelable.Creator<Decimal> CREATOR = new Parcelable.Creator<Decimal>() {
    @Override
    public Decimal createFromParcel(Parcel in) {
      return new Decimal(in);
    }

    @Override
    public Decimal[] newArray(int size) {
      return new Decimal[size];
    }
  };

  private Decimal(Parcel in) {
    value = in.readDouble();
    scale = in.readByte();
  }

}
