package com.clover.sdk.internal.util.calc;

import java.math.RoundingMode;

/**
 * Encapsulates math operations on currency amounts.
 */
public final class Price extends Decimal {

  public static final Decimal HUNDRED = new Decimal(100);

  public static final Price ZERO = new Price(0);

  /**
   * Using a fixed scale on numbers whose
   */
  public static Price divideFixedScale(Price dividend, Decimal divisor) {
    return dividend.divide(divisor, 7, RoundingMode.HALF_UP);
  }

  public Price(long cents) {
    super(cents, 2);
  }

  public Price(String amountString) {
    super(amountString);
  }

  public Price(double dbl, int scale) {
    super(dbl, scale);
  }

  public Price(Price pval, int scale) {
    super(pval, scale);
  }

  public Price(Decimal dval, int newScale) {
    super(dval, newScale);
  }

  public long getCents() {
    return multiply(HUNDRED, 0, RoundingMode.HALF_UP).longValue();
  }

  @Override
  public Price multiply(Decimal dval) {
    return new Price(value * dval.value, Math.max(scale, dval.scale));
  }

  @Override
  public Price divide(Decimal dval, int newScale, RoundingMode newRoundingMode) {
    if (newRoundingMode != RoundingMode.HALF_UP) {
      throw new IllegalArgumentException();
    }
    return new Price(value / dval.value, newScale);
  }

  /**
   * Divide by {@code n} and return the result.  The result is not rounded.
   */
  @Override
  public Price divide(Decimal n) {
    return divideFixedScale(this, n);
  }

  /**
   * Add the given price and return the result.  {@code p} is rounded to 2 decimal places
   * to ensure that we don't add fractions of cents.
   */
  public Price add(Price p) {
    Price rounded = round(p);
    return new Price(value + rounded.value, Math.max(scale, rounded.scale));
  }

  /**
   * Subtract the given price and return the result.  {@code p} is rounded to 2 decimal places
   * to ensure that we don't subtract fractions of cents.
   */
  public Price subtract(Price p) {
    Price rounded = round(p);
    return new Price(value - rounded.value, Math.max(scale, rounded.scale));
  }

  @Override
  public Price setScale(int newScale, RoundingMode newRoundingMode) {
    if (newRoundingMode != RoundingMode.HALF_UP) {
      throw new IllegalArgumentException("RoundingMode.HALF_UP is the only accepted rounding mode");
    }

    if (scale == newScale) {
      return this;
    }

    return new Price(this, newScale);
  }

  public boolean isLessThan(Price other) {
    return compareTo(other) < 0;
  }

  public boolean isGreaterThan(Price other) {
    return compareTo(other) > 0;
  }

  public Price round() {
    return setScale(2, RoundingMode.HALF_UP);
  }

  public static Price round(Price n) {
    return n.round();
  }

}
