package com.clover.sdk.internal.util.calc;

class Validator {

  private static final Decimal HUNDRED = new Decimal(100);

  private final Calc.Order order;
  private final Calc.Logger log;

  Validator(Calc.Order order) {
    this(order, null);
  }

  Validator(Calc.Order order, Calc.Logger log) {
    if (order == null) {
      throw new NullPointerException("order cannot be null");
    }
    if (log == null) {
      log = new Calc.Logger() {
        @Override
        public void warn(String s) {
          // Do nothing by default.
        }
      };
    }

    this.order = order;
    this.log = log;
  }

  Price getPrice(Calc.LineItem line) {
    Price price = line.getPrice();
    if (price == null) {
      return Price.ZERO;
    }
    if (!line.allowNegativePrice() && price.isLessThan(Price.ZERO)) {
      log.warn("Negative line item price: " + price + ".  Using 0 instead.");
      return Price.ZERO;
    }
    return price;
  }

  Decimal getSplitPercent(Calc.LineItem line) {
    Decimal percent = line.getSplitPercent();
    if (percent == null) {
      percent = HUNDRED;
    }
    if (percent.compareTo(Decimal.ZERO) < 0 || percent.compareTo(HUNDRED) > 0) {
      log.warn("Split percent value out of range: " + percent + ".  Using 100%.");
      percent = HUNDRED;
    }
    return percent;
  }

  Decimal getUnitQuantity(Calc.LineItem line) {
    Decimal unitQty = line.getUnitQuantity();
    if (unitQty == null) {
      return Decimal.ONE;
    }
    if (unitQty.compareTo(Decimal.ZERO) < 1) {
      log.warn("Negative unit quantity: " + unitQty + ".  Using 0.");
      return Decimal.ZERO;
    }
    return unitQty;
  }

  Price getModification(Calc.LineItem line) {
    return validatePositiveAmount(line.getModification(), "line item modification");
  }

  Decimal getOrderPercentDiscount() {
    return validatePercent(order.getPercentDiscount(), Decimal.ZERO, "order percent discount");
  }

  Decimal getLinePercentDiscount(Calc.LineItem line) {
    return validatePercent(line.getPercentDiscount(), Decimal.ZERO, "line item percent discount");
  }

  Price getOrderAmountDiscount() {
    return validatePositiveAmount(order.getAmountDiscount(), "order amount discount");
  }

  Price getLineAmountDiscount(Calc.LineItem line) {
    return validatePositiveAmount(line.getAmountDiscount(), "line item amount discount");
  }

  Price getComboDiscount() {
    return validatePositiveAmount(order.getComboDiscount(), "order combo discount");
  }

  Decimal getPercentServiceCharge() {
    return validatePercent(order.getPercentServiceCharge(), Decimal.ZERO, "percent service charge");
  }

  Decimal validatePercent(Decimal percent, Decimal defaultPercent, String name) {
    if (percent == null) {
      return defaultPercent;
    }
    if (percent.compareTo(Decimal.ZERO) < 0) {
      log.warn("Negative " + name + ": " + percent + ".  Using 0.");
      return Decimal.ZERO;
    }
    if (percent.compareTo(HUNDRED) > 1) {
      log.warn(name + " is greater than 100: " + percent + ".  Using 100.");
      return HUNDRED;
    }
    return percent;
  }

  Price validatePositiveAmount(Price price, String name) {
    if (price == null) {
      return Price.ZERO;
    }
    if (price.isLessThan(Price.ZERO)) {
      log.warn(name + " cannot be negative: " + price + ".  Using 0.");
      return Price.ZERO;
    }
    return price;
  }
}
