package com.clover.sdk.v3.order;

import android.os.Bundle;
import android.util.Log;
import com.clover.sdk.internal.util.Lists;
import com.clover.sdk.internal.util.calc.Calc;
import com.clover.sdk.internal.util.calc.Decimal;
import com.clover.sdk.internal.util.calc.Price;
import com.clover.sdk.v3.base.ServiceCharge;
import com.clover.sdk.v3.inventory.TaxRate;
import com.clover.sdk.v3.payments.Payment;

import java.util.Collection;
import java.util.Collections;
import java.util.List;

/**
 * Interfaces with {@link Calc} to calculate total, discount, and
 * tax of an {@link Order}.
 */
public class OrderCalc {

  private static final String TAG = OrderCalc.class.getSimpleName();
  private static final Decimal TAX_RATE_DIVISOR = new Decimal(100000);
  private static final Decimal UNIT_QUANTITY_DIVISOR = new Decimal(1000);
  private static final Decimal HUNDRED = new Decimal(100);

  Order order;
  CalcOrder calcOrder;

  private final Calc.Logger logger = new Calc.Logger() {
    @Override
    public void warn(String s) {
      Log.w(TAG, s);
    }
  };

  private class CalcOrder implements Calc.Order {
    @Override
    public boolean isVat() {
      return order.isNotNullIsVat() && order.getIsVat();
    }

    @Override
    public boolean isTaxRemoved() {
      return order.isNotNullTaxRemoved() ? order.getTaxRemoved() : false;
    }

    @Override
    public Collection<Calc.LineItem> getLineItems() {
      if (order.getLineItems() == null) {
        return Collections.emptyList();
      }
      List<Calc.LineItem> calcLines = Lists.newArrayList();
      for (LineItem line : order.getLineItems()) {
        calcLines.add(new CalcLineItem(line));
      }
      return calcLines;
    }

    @Override
    public Price getComboDiscount() {
      //long discount = -sumOfAdjustments(order.getComboAdjustments(), Adjustment.AdjustmentType.COMBO_DISCOUNT);
      return new Price(0);
    }

    @Override
    public Price getAmountDiscount() {
      long discount = -sumOfAdjustments(order.getDiscounts());
      return new Price(discount);
    }

    @Override
    public Decimal getPercentDiscount() {
      return OrderCalc.getPercentDiscount(order.getDiscounts());
    }

    @Override
    public Price getTip() {
      long totalTips = 0;
      if (order.isNotNullPayments()) {
        for (Payment p : order.getPayments()) {
          if (p.isNotNullTipAmount()) {
            totalTips += p.getTipAmount();
          }
        }
      }
      return new Price(totalTips);
    }

    @Override
    public Decimal getPercentServiceCharge() {
      ServiceCharge sc = order.getServiceCharge();
      if (sc == null || sc.getPercentage() == null) {
        return Decimal.ZERO;
      }
      return new Decimal(sc.getPercentage());
    }
  }

  private class CalcLineItem implements Calc.LineItem {

    LineItem line;

    CalcLineItem(LineItem line) {
      if (line == null) {
        throw new NullPointerException("line cannot be null");
      }
      this.line = line;
    }

    @Override
    public Price getPrice() {
      return new Price(line.getPrice());
    }

    @Override
    public Decimal getUnitQuantity() {
      if (line.getUnitQty() == null) {
        return null;
      }
      return new Decimal(line.getUnitQty()).divide(UNIT_QUANTITY_DIVISOR);
    }

    @Override
    public boolean isRefunded() {
      return line.isNotNullRefunded() && line.getRefunded();
    }

    @Override
    public boolean isExchanged() {
      return line.isNotNullExchanged() && line.getExchanged();
    }

    @Override
    public Collection<Decimal> getTaxRates() {
      List<Decimal> taxRates = Lists.newArrayList();
      if (line.isNotNullTaxRates()) {
        for (TaxRate tax : line.getTaxRates()) {
          if (tax.getRate() != null) {
            taxRates.add(new Decimal(tax.getRate()).divide(TAX_RATE_DIVISOR));
          }
        }
      }
      return taxRates;
    }

    @Override
    public Price getModification() {
      if (!line.isNotNullModifications()) {
        return Price.ZERO;
      }
      long modTotal = 0;
      for (Modification m : line.getModifications()) {
        if (m.getAmount() != null) {
          modTotal += m.getAmount();
        }
      }
      return new Price(modTotal);
    }

    @Override
    public Price getAmountDiscount() {
      long discount = -sumOfAdjustments(line.getDiscounts());
      return new Price(discount);
    }

    @Override
    public Decimal getPercentDiscount() {
      return OrderCalc.getPercentDiscount(line.getDiscounts());
    }

    @Override
    public boolean allowNegativePrice() {
      return order.isNotNullManualTransaction() && order.getManualTransaction();
    }

    public static final String KEY_PERCENT = "percent";

    @Override
    public Decimal getSplitPercent() {
      // TODO: consider removing when making this sdk public - used for transient calculations
      Bundle bundle = line.getBundle();
      LineItemPercent lineItemPercent = bundle.getParcelable(KEY_PERCENT);
      if (lineItemPercent != null) {
        return lineItemPercent.percent.multiply(HUNDRED);
      }
      return HUNDRED;
    }
  }

  public OrderCalc(Order order) {
    this.order = order;
    this.calcOrder = new CalcOrder();
  }

  private Calc getCalc() {
    return new Calc(calcOrder, logger);
  }

  private static long sumOfAdjustments(Collection<Discount> discounts) {
    if (discounts == null) {
      return 0;
    }

    long total = 0;
    for (Discount d : discounts) {
      if (d.getAmount() != null) {
        total += d.getAmount();
      }
    }
    return total;
  }

  private static Decimal getPercentDiscount(Collection<Discount> discounts) {
    if (discounts == null) {
      return Decimal.ZERO;
    }
    long percentTotal = 0;
    for (Discount d : discounts) {
      if (d.getPercentage() != null) {
        percentTotal += d.getPercentage();
      }
    }
    if (percentTotal < 0) {
      percentTotal = 0;
    }
    if (percentTotal > 100) {
      percentTotal = 100;
    }
    return new Decimal(percentTotal);
  }

  public long getTax() {
    return getCalc().getTax().getCents();
  }

  public long getTax(Collection<LineItem> lines) {
    if (lines == null) {
      return 0;
    }
    Price tax = getCalc().getTax(toCalcLines(lines));
    return tax.getCents();
  }

  public long getTip() {
    return calcOrder.getTip().getCents();
  }

  public long getTotal(Collection<LineItem> lines) {
    return getCalc().getTotal(toCalcLines(lines)).getCents();
  }

  private List<CalcLineItem> toCalcLines(Collection<LineItem> lines) {
    List<CalcLineItem> calcLines = Lists.newArrayList();
    if (lines != null) {
      for (LineItem line : lines) {
        calcLines.add(new CalcLineItem(line));
      }
    }
    return calcLines;
  }

  public long getDiscountedSubtotal(Collection<LineItem> lines) {
    return getCalc().getDiscountedSubtotal(toCalcLines(lines)).getCents();
  }

  public long getLineSubtotal(Collection<LineItem> lines) {
    return getCalc().getLineSubtotal(toCalcLines(lines)).getCents();
  }

  public long getServiceCharge() {
    return getCalc().getServiceCharge().getCents();
  }

  public long getServiceCharge(Collection<LineItem> lines) {
    return getCalc().getServiceCharge(toCalcLines(lines)).getCents();
  }

  public long getTotalWithTip(Collection<LineItem> lines) {
    Price total = getCalc().getTotal(toCalcLines(lines));
    return total.add(calcOrder.getTip()).getCents();
  }

  public List<Calc.TaxSummary> getTaxSummaries(Collection<LineItem> lines) {
    return getCalc().getTaxSummaries(toCalcLines(lines));
  }

  public Decimal getDiscountMultiplier() {
    return getCalc().getDiscountMultiplier();
  }

  public Calc.PaymentDetails getPaymentDetails(long paymentAmount) {
    return getCalc().getPaymentDetails(new Price(paymentAmount));
  }

  public Calc.PaymentDetails getPaymentDetails(long paymentAmount, Collection<LineItem> lines) {
    return getCalc().getPaymentDetails(new Price(paymentAmount), toCalcLines(lines));
  }
}
