package codacy.git

import codacy.foundation.logging.context.{ContextLogging, ProjectLogContext}
import org.apache.commons.io.FileUtils

import java.io.{File, InputStream}
import java.nio.charset.CodingErrorAction
import java.util.{Timer, TimerTask}
import scala.concurrent.duration.Duration
import scala.io.Codec
import scala.sys.process._
import scala.util.Try
import scala.util.control.NonFatal

object GitCommandRunner extends ContextLogging {

  /**
    * Executes a git command and returns the output as T.
    * Throws Exception if retries are exhausted.
    *
    * @param command the command to execute
    * @param directory the directory where the command should be executed
    * @param timeout the timeout for the command
    * @param zero the initial value for the accumulator
    * @param fn the function to process the output of the command
    */
  protected[git] def exec[T](command: Seq[String], directory: Option[File] = None, timeout: Option[Duration] = None)(
      zero: T,
      fn: (T, String) => T
  )(implicit logContext: ProjectLogContext): T = {
    runBfs(command, directory, timeout.getOrElse(Duration.Inf))(zero, fn)
  }

  protected[git] def execNoOutput(
      command: Seq[String],
      directory: Option[File] = None,
      timeout: Option[Duration] = None
  )(implicit logContext: ProjectLogContext): Unit = {
    runBfs[Unit](command, directory, timeout.getOrElse(Duration.Inf))((), (_, _) => ())
  }

  private val codec: Codec = {
    val codec0 = Codec.UTF8
    codec0.onMalformedInput(CodingErrorAction.IGNORE)
    codec0.onUnmappableCharacter(CodingErrorAction.IGNORE)
    codec0
  }

  private class OnlyNewLineBufferedSource(inputStream: InputStream)
      extends scala.io.BufferedSource(inputStream)(codec) {
    // tweaking the getLines method to get backward compatible lines
    private class OnlyNewLineIterator extends LineIterator {
      override def isNewline(ch: Char): Boolean = ch == '\n'
    }

    override def getLines(): Iterator[String] = new OnlyNewLineIterator()
  }

  private def triggerTimeout(process: Process, timeout: Duration): Timer = {
    val timer = new Timer()

    if (timeout != Duration.Inf) {
      timer.schedule(new TimerTask {
        def run(): Unit = {
          Try { process.destroy() }
        }
      }, timeout.toMillis)
    }

    timer
  }

  private trait Error
  private case object IndexLockError extends Error
  private case object FatalError extends Error
  private case object UnadvertisedObjectRequest extends Error

  private def parseStdErr(line: String): Option[Error] = {
    if (line.contains("index.lock")) {
      Some(IndexLockError)
    } else if (line.contains("fatal: Invalid symmetric difference expression")) {
      Some(FatalError)
    } else if (line.contains("Server does not allow request for unadvertised object")) {
      Some(UnadvertisedObjectRequest)
    } else {
      None
    }
  }

  private def runBfs[T](cmd: Seq[String], dir: Option[File], timeout: Duration, retries: Int = 1)(
      zero: T,
      fn: (T, String) => T
  )(implicit logContext: ProjectLogContext): T = {
    val commandLog = cmd.mkString("`", " ", "`")
    val dirLog = dir.map(_.getPath).getOrElse("N/A")

    logger.info(s"Executing git command: $commandLog, pwd: $dirLog, timeout: $timeout, retries: $retries")

    var commandResult: T = zero
    var commandErrorText: Seq[String] = Seq()

    var userCodeThrowable = Option.empty[Throwable]

    val pio = new ProcessIO(
      writeInput => { writeInput.close() },
      stdout => {
        val buffSource = new OnlyNewLineBufferedSource(stdout)

        try {
          commandResult = buffSource.getLines().foldLeft(zero)(fn)
        } catch {
          case NonFatal(e) =>
            userCodeThrowable = Some(e)
        } finally {
          stdout.close()
        }

      },
      stderr => {
        val buffSource = new OnlyNewLineBufferedSource(stderr)

        commandErrorText = buffSource
          .getLines()
          .toList

        stderr.close()
      }
    )

    val process: Process = Process(cmd, dir).run(pio)

    val timeoutTimer = triggerTimeout(process, timeout)

    val result = process.exitValue()

    userCodeThrowable.foreach { throw _ }

    timeoutTimer.cancel()

    if (result == 0) {
      commandResult
    } else {
      logger.warn(s"Command failed: $commandLog, pwd: $dirLog, result: $result")

      Try { process.destroy() }

      val commandErrors: Seq[Error] = commandErrorText.flatMap(line => {
        parseStdErr(line)
      })

      retries > 0 match {
        case true if commandErrors.contains(IndexLockError) =>
          logger.warn(
            s"IndexLockError error while executing command: $commandLog, pwd: $dirLog. " +
              s"Delete the lock file and retry."
          )

          Try {
            FileUtils.forceDelete(new java.io.File(dir + java.io.File.separator + ".git", "index.lock"))
          }

          runBfs(cmd, dir, timeout, retries - 1)(zero, fn)

        case true if commandErrors.contains(FatalError) =>
          logger.warn(
            s"Fatal error while executing command: $commandLog, pwd: $dirLog. " +
              s"Run a recovery command and repeat."
          )

          val recoveryCmd = Seq("git", "clean", "-dfx")
          runBfs[Unit](recoveryCmd, dir, timeout, 0)((), (_, _) => ())
          runBfs(cmd, dir, timeout, retries - 1)(zero, fn)

        case true if commandErrors.contains(UnadvertisedObjectRequest) =>
          logger.warn(s"UnadvertisedObjectRequest error while executing command: $commandLog, pwd: $dirLog. Retry.")

          // Works around git-fetch failure on repository with submodules as described in CY-767
          runBfs(cmd, dir, timeout, retries - 1)(zero, fn)

        case _ =>
          commandErrorText.foreach(line => {
            logger.error(line)
          })

          val errorStr = s"Error running command: $commandLog, pwd: $dirLog, result: $result"
          logger.error(errorStr)

          throw new Exception(errorStr)
      }
    }
  }
}
