package codacy.git.authentication

import codacy.foundation.logging.context.{ContextLogging, ProjectLogContext}
import codacy.git.repository.{GitRepositoryDefinition, ProjectRequest}
import play.api.libs.json.Json

import java.io.{File, FileWriter}
import scala.util.control.NonFatal
import scala.util.{Failure, Success, Try}

object SSHAuthenticationWrapper extends ContextLogging {

  /**
    * Wrap given command with `ssh-agent` and `ssh-add` command with a private key.
    * WARNING: Include side effects of writing the private key to a file.
    */
  def wrap(projectRequest: ProjectRequest, command: String, repositoryKeysLocation: String)(
      implicit logContext: ProjectLogContext
  ): Seq[String] = {
    val keyFile = projectRequest.hashCode.toString
    val pkeyAddCommand =
      parseProjectData(projectRequest.data.value)
        .flatMap { rep =>
          writeConfigFile(repositoryKeysLocation, keyFile, rep.privateKey)
        }
        .fold("")(keyFile => s"ssh-add ${keyFile.getAbsolutePath} &>/dev/null;")

    Seq("ssh-agent", "bash", "-c", s"($pkeyAddCommand $command)")
  }

  /**
    * Writes a given content to a file with a specified name in a given path.
    *
    * If the file or the directory does not exist, it creates them. After writing the content, it sets the file to be
    * readable and writable for the owner.
    *
    * @param path    the directory path where the file is to be written
    * @param name    the name of the file to write
    * @param content the content to write into the file
    * @return an [[Option]] containing the [[File]] object if the operation is successful or [[None]] if it fails.
    */
  private def writeConfigFile(path: String, name: String, content: String)(
      implicit logContext: ProjectLogContext
  ): Option[File] = {
    Try {
      logger.info("Processing config file")

      val sourceFolder = new File(path)
      if (!sourceFolder.exists()) {
        sourceFolder.mkdirs()
      }

      val file = new File(sourceFolder, name)
      if (!file.exists()) {
        logger.info("Writing a config file")
        val fw = new FileWriter(file)

        fw.write(content)
        fw.flush()
        fw.close()

        file.setReadable(false, false)
        file.setReadable(true)
        file.setWritable(false, false)
        file.setWritable(true)
      }

      logger.info("Config file processed")
      Some(file)
    } match {
      case Success(file) =>
        file
      case Failure(NonFatal(e)) =>
        logger.error(s"Error writing key file: $name", e)
        None
    }
  }

  /**
    * Generates a sequence of strings representing a command that includes an `ssh-add` command with a private key,
    * which is intended to be run in a new shell started by ssh-agent.
    *
    * Side-effect: the private key is written to a file at a given location.
    *
    * @param projectData            the data of the project, it's used to derive the key file name via its hash code.
    *                               it's also parsed to extract repository details which include the private key
    * @param command                the command to execute after the ssh key has been added
    * @param repositoryKeysLocation the location where the private key file is written
    * @return a sequence of strings representing the full command
    */
  def commandWithKey(projectData: String, command: String, repositoryKeysLocation: String)(
      implicit logContext: ProjectLogContext
  ): Seq[String] = {
    val keyFile = projectData.hashCode.toString
    val pkeyAddCommand =
      parseProjectData(projectData)
        .flatMap { rep =>
          writeConfigFile(repositoryKeysLocation, keyFile, rep.privateKey)
        }
        .fold("")(keyFile => s"ssh-add ${keyFile.getAbsolutePath} &>/dev/null;")

    Seq("ssh-agent", "bash", "-c", s"($pkeyAddCommand $command)")
  }

  private[git] def parseProjectData(json: String): Option[GitRepositoryDefinition] = {
    Try(Json.parse(json).asOpt[GitRepositoryDefinition]).toOption.flatten
      .map(repDef => GitRepositoryDefinition(repDef.publicKey, repDef.privateKey))
  }
}
