package codacy.git.diff

import codacy.foundation.json.JsonEnumeration
import play.api.libs.functional.syntax._
import play.api.libs.json.Reads._
import play.api.libs.json._

import scala.util.Try

case class CommitDiff(data: String, diffs: Try[Seq[Diff]])

case class Diff(oldFile: String, newFile: String, op: FileOperation, chunks: List[ChangeChunk]) {
  val isRename: Boolean = oldFile != newFile
}

object Diff {

  // file names have "\t as prefix and " as suffix, need to drop them to compare
  def apply(files: (String, String), op: FileOperation, chunks: List[ChangeChunk]): Diff = {
    val FilePatternRegex = """"\\t(.*)"""".r

    def strip(s: String): String = s match {
      case FilePatternRegex(fileName) => fileName
      case _ => s
    }

    val (oldFile, newFile) = files
    new Diff(strip(oldFile), strip(newFile), op, chunks)
  }

  implicit lazy val fmt: OFormat[Diff] = {
    implicit val r2: Format[RangeInformation] = Json.format[RangeInformation]
    implicit val r1: Format[ChangeChunk] = Json.format[ChangeChunk]

    (__ \ "oldFile")
      .format[String]
      .and((__ \ "newFile").format[String])
      .and((__ \ "op").format[FileOperation])
      .and((__ \ "chunks").format[List[ChangeChunk]])(caseApply, unlift(Diff.unapply))
  }

  private[this] val caseApply =
    Diff.apply(_: String, _: String, _: FileOperation, _: List[ChangeChunk])

}

sealed trait FileOperation

object FileOperation {

  private lazy val updatedFileMode: Int = -1

  def apply(mode: Int, fileOperationType: FileOperationType.Value): FileOperation =
    fileOperationType match {
      case FileOperationType.NewFile => NewFile(mode)
      case FileOperationType.DeletedFile => DeletedFile(mode)
      case FileOperationType.UpdatedFile => UpdatedFile()
    }

  def unapply(fileOperation: FileOperation): (Int, FileOperationType.Value) =
    fileOperation match {
      case NewFile(mode) => (mode, FileOperationType.NewFile)
      case DeletedFile(mode) => (mode, FileOperationType.DeletedFile)
      case UpdatedFile() => (updatedFileMode, FileOperationType.UpdatedFile)
    }

  implicit lazy val fmt: OFormat[FileOperation] =
    (__ \ "mode").format[Int].and((__ \ "type").format[FileOperationType.Value])(apply, unapply)
}

final case class NewFile(mode: Int) extends FileOperation

final case class DeletedFile(mode: Int) extends FileOperation

final case class UpdatedFile() extends FileOperation

object FileOperationType extends Enumeration with JsonEnumeration {
  val NewFile, DeletedFile, UpdatedFile = Value
}

private[git] object LineType extends JsonEnumeration {
  val ContextLine, LineRemoved, LineAdded = Value
}

sealed trait LineChange {
  val lineType: LineType.Value = LineType.withName(this.getClass.getSimpleName)

  def line: String
}

object LineChange {

  def apply(line: String, value: LineType.Value): LineChange =
    (value match {
      case LineType.ContextLine => ContextLine.apply _
      case LineType.LineRemoved => LineRemoved.apply _
      case LineType.LineAdded => LineAdded.apply _
    })(line)

  def unapply(lineChange: LineChange): (String, LineType.Value) =
    lineChange match {
      case ContextLine(line) => (line, LineType.ContextLine)
      case LineRemoved(line) => (line, LineType.LineRemoved)
      case LineAdded(line) => (line, LineType.LineAdded)
    }

  implicit val fmt: OFormat[LineChange] =
    (__ \ "line").format[String].and((__ \ "lineType").format[LineType.Value])(apply, unapply)
}

final case class ContextLine(line: String) extends LineChange

final case class LineRemoved(line: String) extends LineChange

final case class LineAdded(line: String) extends LineChange

case class RangeInformation(oldOffset: Int, oldLength: Int, newOffset: Int, newLength: Int)

case class ChangeChunk(rangeInformation: RangeInformation, changeLines: List[LineChange])
