package codacy.git.diff

import scala.util.Try
import fastparse._
import NoWhitespace._

object GitDiffParser {

  def parse(str: String): Try[Seq[Diff]] = {
    val cleanStr = str.replace("\n\\ No newline at end of file\n", "\n")

    // parseAll will fail with "end-of-input" if \r appears alone, without \n (https://codacy.atlassian.net/browse/CY-1654)
    // git diff command adds a space to the beginning of each non changed line on the diff
    // when the file has \r as newline character, git diff command doesn't consider it as newline
    // so it will not add that space to the beginning of that
    // our parser is expecting this empty space to exist as it is something that git diff uses to identify non changed lines
    // as consequence, when one file has a \r as newline, the parser will fail with msg "end-of-input"
    val result = fastparse.parse(cleanStr, Parsers.allDiffs(_), verboseFailures = true)
    result match {
      case f: Parsed.Failure => scala.util.Failure(new Exception(f.label))
      case Parsed.Success(r, _) => scala.util.Success(r)
    }
  }

  private object Parsers {
    def allDiffs[$: P]: P[Seq[Diff]] = gitDiff.rep(1) ~ End

    def gitDiff[$: P]: P[Diff] =
      P(
        filesChanged ~ mainFileOperation ~ fileOperation.? ~ index.? ~ (textChunks | binaryChunks).? ~ """\ No newline at end of file""".?
      ).map {
        case (oldFile, newFile, op, op2, index, chunks) =>
          Diff((oldFile, newFile), op, chunks.getOrElse(List.empty))
      }

    def filesChanged[$: P]: P[(String, String)] =
      P("diff --git " ~ filename ~ (" " ~ filename) ~ newline).map {
        case (f1, f2) => (f1, f2)
      }

    def mainFileOperation[$: P]: P[FileOperation] =
      fileOperation.?.map(_.getOrElse(UpdatedFile()))

    def fileOperation[$: P]: P[FileOperation] =
      P(deletedFileMode | newFileMode | fileModeChange | copyFile | renameFile)

    def deletedFileMode[$: P]: P[DeletedFile] =
      P("deleted file mode " ~ mode ~ newline).map { m =>
        DeletedFile(m)
      }

    def newFileMode[$: P]: P[NewFile] =
      P("new file mode " ~ mode ~ newline).map { m =>
        NewFile(m)
      }

    def fileModeChange[$: P]: P[NewFile] =
      P("old mode " ~ mode ~ newline ~ "new mode " ~ mode ~ newline).map {
        case (_, m) =>
          NewFile(m)
      }

    def copyFile[$: P]: P[NewFile] =
      P(
        "similarity index " ~ percentage ~ newline ~
          "copy from " ~ line ~
          "copy to " ~ line
      ).map {
        case (m, _, _) =>
          NewFile(m)
      }

    def renameFile[$: P]: P[NewFile] =
      P(
        "similarity index " ~ percentage ~ newline ~
          "rename from " ~ line ~
          "rename to " ~ line
      ).map {
        case (m, _, _) =>
          NewFile(m)
      }

    def digit[$: P] = P(CharIn("0-9"))
    def hash[$: P]: P[String] = P((digit | CharIn("a-f")).rep(1).!)

    def mode[$: P]: P[Int] =
      P(digit.rep(exactly = 6).!.map(_.toInt))

    def percentage[$: P]: P[Int] =
      P(digit.rep(1).! ~ "%").map {
        _.dropRight(1).toInt
      }

    def binaryChunks[$: P]: P[List[ChangeChunk]] =
      P("GIT binary patch" ~ newline ~ binaryChangeChunk.rep(1) | binaryDiff).map(_.toList)

    def textChunks[$: P]: P[List[ChangeChunk]] =
      P(oldFile ~ newFile ~ textChangeChunk.rep(1).?).map {
        case (_, _, seq) => seq.getOrElse(Nil).toList
      }

    def index[$: P] =
      P(("index " ~ hash ~ ".." ~ hash) ~ (" " ~ mode).? ~ newline)

    def oldFile[$: P]: P[String] = P("--- " ~ filename ~ newline)

    def newFile[$: P]: P[String] = P("+++ " ~ filename ~ newline)

    def filenamePart[$: P]: P[String] =
      P(CharsWhile {
        case '"' | '\\' => false
        case _ => true
      }.!)

    def filename[$: P]: P[String] =
      P(("\"\\t" ~ filenamePart ~ ("\\" ~ AnyChar ~ filenamePart).rep ~ "\"").! | P("/dev/null").map(_ => ""))

    def nonEmptyLine[$: P]: P[String] = P(CharsWhile(_ != '\n').! ~ newline)

    def line[$: P]: P[String] = P(CharsWhile(_ != '\n').?.! ~ newline)

    def binaryDiff[$: P]: P[List[ChangeChunk]] =
      P("Binary files " ~ filename ~ " and " ~ filename ~ " differ" ~ newline).map {
        case _ => List.empty
      }

    def binaryChangeChunk[$: P]: P[ChangeChunk] =
      P(("delta " | "literal ") ~ number ~ newline ~ newline.? ~ binaryLine.rep(1) ~ newline).map {
        case (pos, lines) =>
          ChangeChunk(RangeInformation(pos, pos, pos, pos), lines.toList)
      }

    def binaryLine[$: P]: P[ContextLine] = P(nonEmptyLine).map { l =>
      ContextLine(l)
    }

    def textChangeChunk[$: P]: P[ChangeChunk] =
      P(rangeInformation ~ (contextLine | newline) ~ lineChange.rep(1)).map {
        case (ri, opCtx, lines) =>
          /*
           * opCtx should not be added as a line
           * it is the line before the diff starts and
           * if you consider it to be part of the diff it will
           * make the line numbers wrong
           */
          ChangeChunk(ri, lines.toList)
      }

    def rangeInformation[$: P]: P[RangeInformation] =
      P(("@@ -" ~ number) ~ ("," ~ number).? ~ (" +" ~ number) ~ ("," ~ number).? ~ " @@").map {
        case (a, b, c, d) =>
          RangeInformation(a, b.getOrElse(0), c, d.getOrElse(0))
      }

    def lineChange[$: P]: P[LineChange] =
      P(contextLine | addedLine | deletedLine)

    def contextLine[$: P]: P[ContextLine] =
      P(" " ~ line).map { l =>
        ContextLine(l)
      }

    def addedLine[$: P]: P[LineAdded] =
      P("+" ~ line).map { l =>
        LineAdded(l)
      }

    def deletedLine[$: P]: P[LineRemoved] =
      P("-" ~ line).map { l =>
        LineRemoved(l)
      }

    def newline[$: P]: P[Unit] = P(CharIn(" \t").rep ~ "\n")

    def number[$: P]: P[Int] =
      digit.rep(1).!.map(_.toInt)
  }
}
