package codacy.git.repository

import codacy.foundation.logging.context.{ContextLogging, ProjectLogContext}
import codacy.foundation.utils.InputValidation
import codacy.git.authentication.{SSHAuthenticationWrapper, TokenAuthenticationWrapper}
import play.api.libs.json.{Format, Json}

import java.net.URI
import scala.concurrent.ExecutionContext

case class GitRepositoryDefinition(publicKey: String, privateKey: String)

object GitRepositoryDefinition {
  implicit val gitRepositoryDefinitionFmt: Format[GitRepositoryDefinition] =
    Json.format[GitRepositoryDefinition]
}

abstract class GitRepository(repositoryKeysLocation: String)(implicit executionContext: ExecutionContext)
    extends ContextLogging {

  protected def wrapCommandWithAuthorization(request: ProjectRequest, gitCommand: collection.Seq[String])(
      implicit logContext: ProjectLogContext
  ): Seq[String] = {
    val singleStringCommand = gitCommand.mkString("\"", "\" \"", "\"")
    request.data match {
      case SshRepositoryData(_) => SSHAuthenticationWrapper.wrap(request, singleStringCommand, repositoryKeysLocation)
      case AccessTokenRepositoryData(_) => TokenAuthenticationWrapper.wrap(request, singleStringCommand)
      case EmptyRepositoryData => Seq("bash", "-c", s"($singleStringCommand)")
    }
  }

  protected def processProjectUrl(project: ProjectRequest)(implicit logContext: ProjectLogContext): String = {
    val sanitizedUrl = InputValidation.sanitizeCommandParam(project.url)

    project.data match {
      case AccessTokenRepositoryData(_) =>
        RepositoryUrlHelper.toHttpsWithToken(sanitizedUrl, project.data.value).toString
      case _ => sanitizedUrl
    }
  }
}
