package codacy.git.repository

import codacy.foundation.logging.Logger
import codacy.foundation.logging.context.ProjectLogContext
import codacy.foundation.logging.context.Util.renderWithLimit
import codacy.foundation.utils.InputValidation
import codacy.git.runners.CommandRunner

import scala.collection.mutable
import scala.concurrent.duration.Duration
import scala.concurrent.{ExecutionContext, Future}
import scala.util.Success

class ReadOnlyGitRepository(repositoryKeysLocation: String)(implicit executionContext: ExecutionContext)
    extends GitRepository(repositoryKeysLocation) {

  import codacy.git.repository.ReadOnlyGitRepository._

  /**
    * Retrieves all branches and PR refs, as well as the ref of the main branch.
    */
  def getRefs(project: ProjectRequest, timeout: Duration)(implicit logContext: ProjectLogContext): Future[GitRefs] = {
    getMatchingRefs(project, timeout, Set(DefaultRefsPattern, GitHubPullRequestRefsPattern, MainPattern))
  }

  /**
    * Retrieves requested branches' and PRs' refs, as well as the ref of the main branch.
    * If no branches or PRs were requested, returns an empty response.
    */
  def getSelectedAndMainRefs(
      project: ProjectRequest,
      branchNames: Set[String],
      prNumbers: Set[Long],
      timeout: Duration
  )(implicit logContext: ProjectLogContext): Future[GitRefs] = {
    val branchRefs = branchNames.map(toBranchRef)
    val gitHubPullRequestRefs = prNumbers.map(toGitHubPullRequestRef)

    branchRefs ++ gitHubPullRequestRefs match {
      case refs if refs.nonEmpty =>
        logger.info(s"Selective branch refs ${refs.mkString(" ,")}")
        getMatchingRefs(project, timeout, refs + MainPattern)

      case _ =>
        logger.warn("No branches or PRs to select")
        Future.successful(GitRefs(Seq.empty, Seq.empty))
    }
  }

  /**
    * Retrieves refs that match requested patterns and parses the result to `ParsedRefState`.
    */
  private def getMatchingRefs(project: ProjectRequest, timeout: Duration, patterns: Set[String])(
      implicit logContext: ProjectLogContext
  ): Future[GitRefs] = {
    val url = processProjectUrl(project)

    val command = Seq("git", "ls-remote", "--symref", url) ++ patterns
    val commandWithAuth = wrapCommandWithAuthorization(project, command)

    CommandRunner
      .runCommand(commandWithAuth, timeout)
      .andThen {
        case Success(result) => logger.info(s"Command returned ${result.size} lines, will process output now")
      }
      .map(refs => refs.foldLeft(ParsedRefState())(parseRef).getRefs)
  }
}

object ReadOnlyGitRepository extends Logger {
  val prPrefix = "codacy-amN6znLRueK4Yky"

  val MainPattern = "HEAD"
  val DefaultRefsPattern = "refs/heads/*"

  /**
    * Only GitHub supports these pull request refs. We fetch them to pull in the PRs from the
    * forked repositories. That feature is only supported for GitHub. For more info see IO-713.
    */
  private val GitHubPullRequestRefsPattern = "refs/pull/*/head"

  private val MainBranchName = """^ref: refs/heads/(.*)[\t]+HEAD$""".r
  private val MainBranchUUID = """^([a-z0-9]+)[ \t]+HEAD$""".r
  private val Branch = """^([a-z0-9]+)[ \t]+refs/heads/(.*)$""".r
  private val GitHubPullRequest = """^([a-z0-9]+)[ \t]+refs/pull/([0-9]+)/head$""".r

  /**
    * These represent invalid branch names that are reserved for git. Branches with these names can be created
    * in some provider's UIs, but are not allowed by the git CLI.
    */
  private val InvalidBranch = """^([a-z0-9]+)[ \t]+refs/heads/(?:HEAD|refs/heads/.*)$""".r

  private def toBranchRef(branchName: String): String = s"refs/heads/$branchName"

  private def toGitHubPullRequestRef(prNumber: Long): String = s"refs/pull/$prNumber/head"

  protected[git] case class ParsedRefState(
      mainBranchUUID: Option[String] = None,
      mainBranchName: Option[String] = None,
      branchesMapBuilder: mutable.Builder[(String, GitBranch), Map[String, GitBranch]] = Map.newBuilder,
      pullRequestsMapBuilder: mutable.Builder[(String, GitPullRequest), Map[String, GitPullRequest]] = Map.newBuilder,
  ) {

    def getRefs: GitRefs = {
      val branches =
        toSeqWithMain(branchesMapBuilder.result())(mainBranchUUID.getOrElse(""), mainBranchName.getOrElse(""))

      val gitHubPullRequests = pullRequestsMapBuilder.result().valuesIterator.toList

      GitRefs(branches, gitHubPullRequests)
    }
  }

  /** Parses a line to extract branches, pull requests, and the repository's main branch. */
  private[git] def parseRef(state: ParsedRefState, line: String): ParsedRefState = {
    line match {
      case InvalidBranch(latestCommitUUID) =>
        logger.warn(
          s"Failed to insert branch with commit UUID $latestCommitUUID." +
            s"Invalid line: $line"
        )
        state

      case Branch(latestCommitUUID, name) =>
        state.branchesMapBuilder += name -> GitBranch(name, latestCommitUUID)
        state

      case GitHubPullRequest(latestCommitUUID, number) =>
        val name = s"$prPrefix/pr/$number"
        state.pullRequestsMapBuilder += name -> GitPullRequest(name, number.toInt, latestCommitUUID)
        state

      case MainBranchUUID(latestCommitUUID) =>
        state.copy(mainBranchUUID = Option(latestCommitUUID))

      case MainBranchName(name) =>
        state.copy(mainBranchName = Option(name))

      case _ => state
    }
  }

  /**
    * Sets `GitBranch.isMainBranch` to `true` for the main branch
    */
  private def toSeqWithMain(
      allBranches: Map[String, GitBranch]
  )(mainBranchUUID: String, mainBranchName: String): Seq[GitBranch] = {
    val mainBranch = GitBranch(name = mainBranchName, lastCommitUUID = mainBranchUUID, isMainBranch = true)

    // Done to guarantee that mainBranch is only listed once.
    allBranches.foldRight(Seq(mainBranch)) {
      case ((name, branch), acc) =>
        if (name != mainBranchName)
          branch +: acc
        else acc
    }
  }
}
