package codacy.git.repository

import codacy.foundation.logging.context.Util.renderWithLimit
import org.joda.time.DateTime

case class CommitRequest(uuid: String)

/**
  * @param url repository url - can be ssh url or https url
  * @param data SSH keys or access token
  */
case class ProjectRequest(url: String, data: RepositoryData)

case class BranchRequest(name: String)

case class Blame(blameCommit: String, originalLine: Int, newLine: Int, originalFile: String)

/**
  * Holds information about repository authentication data.
  * This can be either an SSH key or an access token.
  */
sealed trait RepositoryData {
  val value: String
}

case class SshRepositoryData(value: String) extends RepositoryData
case class AccessTokenRepositoryData(value: String) extends RepositoryData
case object EmptyRepositoryData extends RepositoryData {
  override val value: String = ""
}

case class GitCommit(
    uuid: String,
    parentUuid: String,
    authorTimestamp: DateTime,
    commitTimestamp: DateTime,
    authorName: String,
    authorEmail: String,
    message: String
)

case class GitCommitWithIndex(commit: GitCommit, index: Int)

case class GitRefs(branches: Seq[GitBranch], pullRequests: Seq[GitPullRequest]) {
  override def toString: String =
    s"GitRefs: [Branches: ${renderWithLimit(branches)}; Pull requests: ${renderWithLimit(pullRequests)}]"
}

case class GitBranch(name: String, lastCommitUUID: String, isMainBranch: Boolean = false)

case class GitPullRequest(name: String, number: Int, lastCommitUUID: String)

case class GitFile(filename: String, hash: String, bytes: Int)

case class GitListResult(files: List[GitFile], bypassed: List[GitFile])

case class GitFileContents(filename: String, lines: List[String], isFileTooBig: Boolean = false)
