package codacy.git.repository

import codacy.foundation.logging.context.{ContextLogging, ProjectLogContext}

import java.net.URI

object RepositoryUrlHelper extends ContextLogging {

  val xAccessToken = "x-access-token"
  val gitRepoRegex = """(git|ssh|https?|git@[-\w.]+):(\/\/)?(.*?)(\.git)?(\/?|\#[-\w._]+?)$""".r
  val gitSshRegex = """(git|ssh|https)@?([-\w.]+):(\/\/)?(.*?)(\.git)?(\/?|\#[-\w._]+?)$""".r

  def getProtocol(url: String): Protocol.Value = {
    gitRepoRegex
      .findAllMatchIn(url)
      .toSeq
      .headOption
      .map { matched =>
        val protocol = matched.group(1)
        protocol match {
          case "https" => Protocol.https
          case p if p.startsWith("git") => Protocol.ssh
          case _ => Protocol.invalid
        }
      }
      .getOrElse {
        Protocol.invalid
      }
  }

  /**
    * Recognizes the protocol of the given url. Then change url to https with given token.
    * Throws IllegalArgumentException if the protocol is not recognized.
    */
  def toHttpsWithToken(url: String, token: String, userPart: String = xAccessToken)(
      implicit logContext: ProjectLogContext
  ): URI = {
    RepositoryUrlHelper.getProtocol(url) match {
      case Protocol.ssh => RepositoryUrlHelper.changeSshUrlToHttpsWithToken(url, token, userPart)
      case Protocol.https => RepositoryUrlHelper.addToken(url, token, userPart)
      case Protocol.invalid =>
        val message = s"Invalid protocol in url: $url"
        logger.error(message)
        throw new IllegalArgumentException(message)
    }
  }

  private def changeSshUrlToHttpsWithToken(url: String, token: String, userPart: String = xAccessToken): URI = {
    val urlString = gitSshRegex.replaceAllIn(url, { matched =>
      val domainPart = matched.group(2)
      val repoPath = matched.group(4)
      s"https://$userPart:$token@$domainPart/$repoPath.git"
    })
    URI.create(urlString)
  }

  private def addToken(url: String, token: String, userPart: String = xAccessToken): URI = {
    val uri = URI.create(url)
    URI.create(s"https://$userPart:$token@${uri.getHost}${uri.getPath}")
  }

}

object Protocol extends Enumeration {
  val ssh = Value
  val https = Value
  val invalid = Value
}
