package codacy.git.repository

import codacy.foundation.logging.context.ProjectLogContext
import codacy.git._
import codacy.utils.FileSystemLocks

import java.nio.file.Paths
import scala.collection.mutable.ListBuffer
import scala.concurrent.ExecutionContext
import scala.util.Try

class WorktreeGitRepository(repositoryKeysLocation: String, repositoryLocation: String)(
    implicit executionContext: ExecutionContext
) extends WriteGitRepository(repositoryKeysLocation, repositoryLocation) {

  private final val commitWithHashRegex = "^[0-9a-f]{40}".r

  override def pathPrefix: String = "p_"

  override protected[git] def withRepository[T](project: ProjectRequest, forceUpdate: Boolean = false)(
      block: java.io.File => T
  )(implicit logContext: ProjectLogContext): T = {
    withRepositoryOptions(project, forceUpdate = forceUpdate)(block)
  }

  def withRepositoryOptions[T](
      project: ProjectRequest,
      commitOpt: Option[CommitRequest] = None,
      forceUpdate: Boolean = false,
      cloneSubmodules: Boolean = false
  )(block: java.io.File => T)(implicit logContext: ProjectLogContext): T = {
    val projectDir = getProjectPath(project.url)

    lock(projectDir) { implicit l: FileSystemLocks.Lock =>
      val repositoryDir = pullChangesOrClone(project, projectDir, forceUpdate, cloneSubmodules)

      commitOpt.collect {
        case commit if !WriteGitRepository.isValidRepo(projectDir, Some(commit.uuid)).getOrElse(false) =>
          checkout(projectDir, project, commit)
      }

      block(repositoryDir)
    }
  }

  def updateRepository(projectRequest: ProjectRequest, cloneSubmodules: Boolean = false)(
      implicit logContext: ProjectLogContext
  ): Unit = {
    val projectDir = getProjectPath(projectRequest.url)

    lock(projectDir) { implicit l: FileSystemLocks.Lock =>
      pullChangesOrClone(projectRequest, projectDir, cloneSubmodules)
    }
  }

  def updateRepository(project: ProjectRequest)(implicit logContext: ProjectLogContext): Unit = {
    updateRepository(project, false)
  }

  def getBlame(dir: java.io.File, filename: String, startLine: Int, endLine: Int)(
      implicit logContext: ProjectLogContext
  ): Option[collection.Seq[Blame]] = {
    val blameFile = Paths.get(dir.getAbsolutePath, filename).normalize().toFile

    if (!blameFile.isFile) {
      logger.warn(s"Blame file not found: ${blameFile.getAbsolutePath}")
      None
    } else {
      val (res, _) =
        GitCommandRunner
          .exec[(ListBuffer[Blame], Option[String])](
            Seq(
              "git",
              "blame",
              "-ls",
              "--root",
              "--line-porcelain",
              s"-L$startLine,$endLine",
              blameFile.getAbsolutePath
            ),
            Some(dir)
          )(
            (ListBuffer[Blame](), None), { (prevRes: (ListBuffer[Blame], Option[String]), current: String) =>
              // only lines that matter for this are "filename ..." and ones that start by the commit hash
              val shouldProcessLine = current
                .startsWith("filename ") || commitWithHashRegex.findFirstIn(current).nonEmpty

              if (shouldProcessLine) {
                val (prev, last) = prevRes
                last match {
                  case None => (prev, Some(current))
                  case Some(p) =>
                    WriteGitRepository.parseBlame(p, current) match {
                      case Some(blame) => (prev += blame, None)
                      case None => (prev, None)
                    }
                }
              } else {
                prevRes
              }
            }
          )
      Some(res)
    }
  }

  def listSubmodulesFiles(dir: java.io.File, commit: String, maxFileSizeBytesOpt: Option[Int] = Option.empty)(
      implicit logContext: ProjectLogContext
  ): Option[GitListResult] = {
    import WriteGitRepository.{isCommit, isSymbolicLink, GitFileResponse}

    Try {
      val normalizedFiles =
        GitCommandRunner
          .exec(Seq("git", "ls-tree", "-l", "-r", commit), Some(dir))(
            List[GitFile](),
            (resps: List[GitFile], line: String) => {
              line match {
                case GitFileResponse(mode, t, hash, _, filename) if !isSymbolicLink(mode) && isCommit(t) =>
                  resps ++
                    listFiles(Paths.get(dir.getAbsolutePath, filename).normalize().toFile, hash, maxFileSizeBytesOpt).files
                case _ => resps
              }
            }
          )
          .distinct

      GitListResult(normalizedFiles, List.empty)
    }.toOption
  }

  private def checkout(dir: java.io.File, project: ProjectRequest, commit: CommitRequest, retries: Int = 0)(
      implicit l: FileSystemLocks.Lock,
      logContext: ProjectLogContext
  ): Option[Unit] = {
    if (retries < 2) {
      reset(dir, commit)
        .flatMap { _ =>
          clean(dir)
        }
        .orElse {
          pullChanges(dir, project)
          checkout(dir, project, commit, retries + 1)
        }
        .orElse {
          logger.error(s"Could not checkout commit ${commit.uuid}")
          Option.empty[Unit]
        }
    } else {
      Option.empty[Unit]
    }
  }

  private def reset(
      directory: java.io.File,
      commit: CommitRequest
  )(implicit l: FileSystemLocks.Lock, logContext: ProjectLogContext): Option[Unit] = {
    Try {
      GitCommandRunner
        .execNoOutput(Seq("git", "reset", "--hard", s"${commit.uuid}"), Some(directory))
    }.toOption
  }

  private def clean(
      directory: java.io.File
  )(implicit l: FileSystemLocks.Lock, logContext: ProjectLogContext): Option[Unit] = {
    Try {
      GitCommandRunner.execNoOutput(Seq("git", "clean", "-dfx"), Some(directory))
    }.toOption
  }
}
