package codacy.git.repository

import codacy.foundation.crypto.{CryptoTools, SSHKeyGenerator}
import codacy.foundation.logging.context.{ContextLogging, ProjectLogContext}
import codacy.foundation.utils.InputValidation
import codacy.git._
import codacy.git.authentication.{SSHAuthenticationWrapper, TokenAuthenticationWrapper}
import codacy.git.diff.{CommitDiff, GitDiffParser}
import codacy.utils.{FileOperations, FileSystemLocks}
import org.joda.time.DateTime
import play.api.libs.json.Json

import java.io.File
import java.nio.file.Paths
import scala.collection.mutable
import scala.concurrent.ExecutionContext
import scala.concurrent.duration._
import scala.util.control.NonFatal
import scala.util.{Success, Try}

/**
  * Handle all providers and authentication - by ssh and token.
  * Authentication type is recognized and applied basing on given [[ProjectRequest]].
  *
  * @param repositoryKeysLocation - used for ssh access
  */
abstract class WriteGitRepository(val repositoryKeysLocation: String, val repositoryLocation: String)(
    implicit executionContext: ExecutionContext
) extends GitRepository(repositoryKeysLocation)
    with FileSystemLocks {

  import WriteGitRepository._

  protected[repository] def cloneOptions(recursive: Boolean): Seq[String] =
    if (recursive) Seq("--recursive") else Seq.empty

  protected[repository] def pathPrefix: String = "p_"

  protected[git] def withRepository[T](project: ProjectRequest, forceUpdate: Boolean = false)(block: java.io.File => T)(
      implicit logContext: ProjectLogContext
  ): T

  /**
    * Update repository without submodules.
    */
  def updateRepository(projectRequest: ProjectRequest)(implicit logContext: ProjectLogContext): Unit

  protected def pullChangesOrClone(projectReq: ProjectRequest, projectDir: File, cloneSubmodules: Boolean)(
      implicit l: FileSystemLocks.Lock,
      logContext: ProjectLogContext
  ): Unit = {
    logger.info(s"Updating repository. (project: ${projectReq.url} ; cloneSubmodules: $cloneSubmodules)")

    val isValidHeadRsp = WriteGitRepository.isValidRepo(projectDir)
    val isValidHead = isValidHeadRsp.getOrElse(false)

    if (isValidHead) {
      pullChanges(projectDir, projectReq)
    } else {
      cleanAndClone(projectReq, projectDir, cloneSubmodules)
    }
  }

  protected def pullChangesOrClone(
      project: ProjectRequest,
      projectDir: File,
      forceUpdate: Boolean,
      cloneSubmodules: Boolean
  )(implicit l: FileSystemLocks.Lock, logContext: ProjectLogContext): java.io.File = {
    val isValidHeadRsp = WriteGitRepository.isValidRepo(projectDir)
    val isValidHead = isValidHeadRsp.getOrElse(false)

    if (!isValidHead) {
      logger.debug(s"""Forcing removal of HEAD ${projectDir.getAbsolutePath} under conditions:
           |isValidRepo.value.get:$isValidHead
            """.stripMargin)

      cleanAndClone(project, projectDir, cloneSubmodules)
      projectDir
    } else {

      if (forceUpdate) {
        pullChanges(projectDir, project) // only fetch existing folder if requested
      }

      projectDir
    }
  }

  private def cleanAndClone(projectReq: ProjectRequest, projectDir: java.io.File, cloneSubmodules: Boolean)(
      implicit l: FileSystemLocks.Lock,
      logContext: ProjectLogContext
  ): Unit = {
    recreateProjectDir(projectDir)

    try {
      clone(projectDir, projectReq, recursive = cloneSubmodules)
    } catch {
      case NonFatal(err) =>
        recreateProjectDir(projectDir)
        throw err
    }
  }

  def getAllCommits(project: ProjectRequest)(implicit logContext: ProjectLogContext): List[GitCommitWithIndex] = {
    logger.time(s"Getting all commits for project ${project.url}") {
      withRepository(project) { directory =>
        listAllCommits(directory)
      }
    }
  }

  def getBranchCommits(project: ProjectRequest, branch: BranchRequest, limit: Option[Int] = None)(
      implicit logContext: ProjectLogContext
  ): List[GitCommit] = {
    logger.time(s"Getting branch commits for project ${project.url} - ${branch.name}") {
      withRepository(project) { directory =>
        listBranchCommits(directory, project, branch, limit)
      }
    }
  }

  def getBranchCommits(project: ProjectRequest, branch: BranchRequest, limit: CommitRequest)(
      implicit logContext: ProjectLogContext
  ): List[CommitRequest] = {
    logger.time(s"Getting branch commits for project ${project.url} - ${branch.name}") {
      withRepository(project) { directory =>
        listBranchCommits(directory, project, branch, limit)
      }
    }
  }

  /**
    * Fetches a list of Git commits for the specified branches in a given project.
    *
    * @param project  The project request, which includes details about the project such as its URL.
    * @param branches A sequence of branch requests for which commits are to be retrieved.
    * @return A list of GitCommit instances, each representing a commit from the requested branches.
    */
  def getBranchCommits(project: ProjectRequest, branches: Seq[BranchRequest])(
      implicit logContext: ProjectLogContext
  ): List[GitCommit] = {
    logger.time(s"Getting commits for selected branches for project ${project.url}") {
      withRepository(project) { directory =>
        listBranchesCommits(directory, project, branches)
      }
    }
  }

  def findCommonAncestor(project: ProjectRequest, source: BranchRequest, destination: BranchRequest)(
      implicit logContext: ProjectLogContext
  ): Option[CommitRequest] = {
    logger.time(s"Getting common ancestor commit for branches ${project.url} - ${source.name}:${destination.name}") {
      withRepository(project) { directory =>
        findCommonAncestor(directory, source, destination)
      }
    }
  }

  def cleanProjectDir(projectUrl: String)(implicit logContext: ProjectLogContext): Unit = {
    val projectDir = getProjectPath(projectUrl)
    lock(projectDir) { implicit l: FileSystemLocks.Lock =>
      recreateProjectDir(projectDir)
    }
  }

  private def recreateProjectDir(projectDir: File)(implicit logContext: ProjectLogContext): Unit = {
    FileOperations.deleteDirectory(projectDir)
    FileOperations.createDirectory(projectDir)
  }

  private[git] def getProjectPath(iurl: String): java.io.File = {
    Paths
      .get(repositoryLocation)
      .resolve(s"$pathPrefix${CryptoTools.sha256Alphanumeric(iurl).toLowerCase.take(16)}")
      .toFile
  }

  def generateProjectData(values: Map[String, String])(implicit logContext: ProjectLogContext): RepositoryData =
    WriteGitRepository.generateProjectData(values)

  protected def pullChanges(dir: java.io.File, request: ProjectRequest)(
      implicit logContext: ProjectLogContext
  ): Unit = {
    val command = Seq("git", "fetch", "--all", "--prune", "--force")
    val commandWithAuth = wrapCommandWithAuthorization(request, command)
    GitCommandRunner
      .execNoOutput(commandWithAuth, Some(dir))
  }

  private def clone(dir: java.io.File, project: ProjectRequest, recursive: Boolean)(
      implicit l: FileSystemLocks.Lock,
      logContext: ProjectLogContext
  ): Unit = {
    val url = processProjectUrl(project)
    val cmd: Seq[String] = Seq("git", "clone") ++ cloneOptions(recursive) ++ Seq(url, dir.getAbsolutePath)

    // In clone we do NOT wrap for token auth. Token is in the url.
    val projectData = project.data match {
      case AccessTokenRepositoryData(_) => EmptyRepositoryData
      case other => other
    }

    val cloneCommand = wrapCommandWithAuthorization(project.copy(data = projectData), cmd)

    val cloneCmdResponse = GitCommandRunner.execNoOutput(cloneCommand, Some(dir))

    addOriginReferenceConfiguration(dir)
    addGitHubPullRequestConfiguration(dir)
    pullChanges(dir, project)

    cloneCmdResponse
  }

  def listFiles(dir: java.io.File, commit: String, maxFileSizeBytesOpt: Option[Int] = Option.empty)(
      implicit logContext: ProjectLogContext
  ): GitListResult = {
    logger.debug(s"Listing files in ${dir.getAbsolutePath}")

    val files: List[GitFile] =
      GitCommandRunner
        .exec(Seq("git", "ls-tree", "-l", "-r", commit), Some(dir))(
          mutable.ListBuffer[GitFile](),
          (prev: mutable.ListBuffer[GitFile], line: String) => {
            line match {
              case GitFileResponse(mode, t, hash, size, filename) if !isSymbolicLink(mode) && !isCommit(t) =>
                prev += GitFile(filename, hash, size.toInt)
              case _ => prev
            }
          }
        )
        .distinct
        .toList

    maxFileSizeBytesOpt match {
      case None => GitListResult(files, List())
      case Some(maxFileSize) =>
        GitListResult.tupled(files.partition(_.bytes < maxFileSize))
    }
  }

  def changedFiles(dir: java.io.File, currentCommit: String)(
      implicit logContext: ProjectLogContext
  ): collection.Seq[String] = {
    logger.debug(s"Retrieving changed files in ${dir.getAbsolutePath}")

    GitCommandRunner.exec(
      Seq("git", "show", "--first-parent", "--pretty=format:", "--name-only", currentCommit),
      Some(dir)
    )(mutable.ListBuffer[String](), (prev: mutable.ListBuffer[String], line: String) => {
      val trimmed = line.trim

      if (trimmed.nonEmpty) {
        prev += line.trim
      } else {
        prev
      }
    })
  }

  def getCommitDiff(dir: java.io.File, sourceCommit: CommitRequest, destinationCommits: Seq[String], slf: Set[String])(
      implicit logContext: ProjectLogContext
  ): CommitDiff = {
    val diffCommand = destinationCommits.filter(_.trim.nonEmpty) match {
      case parents if parents.length > 1 =>
        diffCommandPrefix ++ Seq(sourceCommit.uuid) ++ destinationCommits
      case parents =>
        parents.lastOption.fold {
          Seq("git", "show", "--format=") ++ diffPrefix ++ Seq(sourceCommit.uuid) ++ diffSuffix(slf)
        } { lastCommit =>
          diffCommandPrefix ++ Seq(lastCommit, sourceCommit.uuid) ++ diffSuffix(slf)
        }
    }

    executeAndParseDiff(diffCommand, dir)
  }

  def getPullRequestDiff(dir: java.io.File, source: CommitRequest, destination: String, slf: Set[String])(
      implicit logContext: ProjectLogContext
  ): CommitDiff = {
    val command = diffCommandPrefix ++ Seq(s"$destination...${source.uuid}") ++ diffSuffix(slf)

    logger.info(s"""Getting pull request diff. cmd:${command.mkString(" ")}""")

    executeAndParseDiff(command, dir)
  }

  def getFileContents(dir: java.io.File, commit: CommitRequest, filename: String)(
      implicit logContext: ProjectLogContext
  ): GitFileContents = {
    val command = Seq("git", "show", s"${commit.uuid}:$filename")

    GitFileContents(
      filename,
      GitCommandRunner
        .exec(command, Some(dir))(
          mutable.ListBuffer[String](),
          (acc: mutable.ListBuffer[String], line: String) => acc += line
        )
        .toList
    )
  }

  def listAllCommits(dir: java.io.File)(implicit logContext: ProjectLogContext): List[GitCommitWithIndex] = {
    val logCmd = Seq("git", "log", "--all", "--encoding=UTF-8", "--date=local", s"--pretty=format:$commitFormat")

    GitCommandRunner
      .exec(logCmd, Some(dir))(
        mutable.ListBuffer[GitCommit](),
        (prev: mutable.ListBuffer[GitCommit], line: String) => {
          parseCommitLog(line) match {
            case Some(commit) =>
              commit +=: prev
            case None => prev
          }
        }
      )
      .zipWithIndex
      .map { case (e: GitCommit, i: Int) => GitCommitWithIndex(e, i) }
      .toList
  }

  private def listBranchCommits(dir: java.io.File, project: ProjectRequest, branch: BranchRequest, limit: Option[Int])(
      implicit logContext: ProjectLogContext
  ): List[GitCommit] = {
    val limitArgs = limit.map(l => Seq("-n", l.toString)).getOrElse(Seq.empty)

    val logCmd = wrapCommandWithAuthorization(
      project,
      Seq(
        "git",
        "log",
        "--encoding=UTF-8",
        "--date=local",
        s"--pretty=format:$commitFormat",
        s"remotes/origin/${branch.name}"
      ) ++ limitArgs
    )

    parseBranchCommits(logCmd, dir)
  }

  private def listBranchesCommits(dir: java.io.File, project: ProjectRequest, branches: Seq[BranchRequest])(
      implicit logContext: ProjectLogContext
  ): List[GitCommit] = {
    val branchNames = branches.map(branch => s"remotes/origin/${branch.name}")

    val logCmd = Seq("git", "log", "--encoding=UTF-8", "--date=local", s"--pretty=format:$commitFormat") ++ branchNames

    parseBranchCommits(logCmd, dir)
  }

  private def listBranchCommits(
      dir: java.io.File,
      project: ProjectRequest,
      branch: BranchRequest,
      limitCommit: CommitRequest
  )(implicit logContext: ProjectLogContext): List[CommitRequest] = {
    val cmd =
      wrapCommandWithAuthorization(
        project,
        Seq("git", "rev-list", s"${limitCommit.uuid}..remotes/origin/${branch.name}")
      )

    GitCommandRunner
      .exec(cmd, Option(dir))(
        mutable.ListBuffer[CommitRequest](),
        (prev: mutable.ListBuffer[CommitRequest], line: String) => {
          prev += CommitRequest(line)
        }
      )
      .toList
  }

  private def parseBranchCommits(logCmd: Seq[String], dir: java.io.File)(
      implicit logContext: ProjectLogContext
  ): List[GitCommit] = {
    GitCommandRunner
      .exec(logCmd, Some(dir))(
        mutable.ListBuffer[GitCommit](),
        (prev: mutable.ListBuffer[GitCommit], line: String) => {
          parseCommitLog(line) match {
            case Some(commit) => prev += commit
            case None => prev
          }
        }
      )
      .toList
  }

  private def findCommonAncestor(dir: java.io.File, source: BranchRequest, destination: BranchRequest)(
      implicit logContext: ProjectLogContext
  ): Option[CommitRequest] = {
    val cmd =
      Seq("git", "merge-base", s"remotes/origin/${source.name}", s"remotes/origin/${destination.name}")

    // isn't this implementation weak?
    GitCommandRunner.exec(cmd, Option(dir))(None, (prev: Option[CommitRequest], line: String) => {
      prev match {
        case None => Some(CommitRequest(line))
        case _ => prev
      }
    })
  }
}

object WriteGitRepository extends ContextLogging {

  val prPrefix = "codacy-amN6znLRueK4Yky"
  val sep = "<codacy:sep>"
  val eol = "<codacy:eol>"
  val commitFormat = s"%H$sep%P$sep%at$sep%ct$sep%an$sep%ae$sep%s$eol"

  private val CommitMatch =
    s"""^([a-z0-9]+)$sep([a-z0-9 ]*)$sep([0-9]+)$sep([0-9]+)$sep((?s).*)$sep((?s).*)$sep((?s).*)$eol""".r

  private val diffFlags = Seq("-M")
  private val diffCommandPrefix = Seq("git", "diff") ++ diffPrefix ++ diffFlags

  private lazy val diffPrefix = Seq("--src-prefix=\t", "--dst-prefix=\t")

  private def diffSuffix(files: Set[String]): List[String] = List("--") ++ files.map(e => s"*$e")

  private[git] lazy val GitFileResponse = """(\w+)\s*(\w+)\s*(\w+)\s*([0-9-]+)\t(.+)""".r

  private[git] def isSymbolicLink(s: String) = s == "120000"

  private[git] def isCommit(s: String) = s == "commit"

  private def addOriginReferenceConfiguration(dir: java.io.File)(implicit logContext: ProjectLogContext): Unit = {
    addGitConfig(dir, "remote.origin.fetch", "+refs/heads/*:refs/remotes/origin/*")
  }

  private def addGitHubPullRequestConfiguration(dir: java.io.File)(implicit logContext: ProjectLogContext): Unit = {
    addGitConfig(
      dir,
      "remote.origin.fetch",
      s"+refs/pull/*/head:refs/remotes/origin/${WriteGitRepository.prPrefix}/pr/*"
    )
  }

  protected def addGitConfig(dir: java.io.File, name: String, value: String)(
      implicit logContext: ProjectLogContext
  ): Unit = {
    GitCommandRunner.execNoOutput(Seq("git", "config", "--add", name, value), Some(dir))
  }

  private[git] def isValidRepo(dir: java.io.File, commitUuid: Option[String] = None)(
      implicit logContext: ProjectLogContext
  ): Option[Boolean] = {
    Try {
      val result =
        GitCommandRunner.exec(Seq("git", "rev-parse", "--verify", "HEAD"), Some(dir))(
          "",
          (res: String, line: String) => res + line
        )

      (commitUuid.isEmpty || commitUuid.contains(result.trim))
    }.toOption
  }

  private[git] def generateProjectData(
      values: Map[String, String]
  )(implicit logContext: ProjectLogContext): RepositoryData = {
    val (generatedPublicKey, generatedPrivateKey) = try SSHKeyGenerator.generateKey()
    catch {
      case NonFatal(e) =>
        logger.error("Error generating ssh key", e)
        ("", "")
    }
    val publicKey = values.getOrElse("publicKey", generatedPublicKey)
    val privateKey = values.getOrElse("privateKey", generatedPrivateKey)

    val json =
      Json.toJson(GitRepositoryDefinition(publicKey, privateKey)).toString()

    SshRepositoryData(json)
  }

  private[git] def parseCommitLog(commitLog: String)(implicit logContext: ProjectLogContext): Option[GitCommit] = {
    commitLog match {
      case cMatch @ CommitMatch(hash, parents, authorDate, commitDate, authorName, authorEmail, subject) =>
        logger.debug(s"RepositoryGit::parseCommitDateUser::s $cMatch")

        val commit = GitCommit(
          hash,
          parents.trim,
          new DateTime(authorDate.toLong * 1000),
          new DateTime(commitDate.toLong * 1000),
          authorName,
          authorEmail,
          subject
        )

        Option(commit)
      case commit =>
        logger.error(s"RepositoryGit::failed to parse commit [$commit]")
        Option.empty[GitCommit]
    }
  }

  final val BlameMatch = """^([0-9a-f]{40})\s([0-9]+)\s([0-9]+)\s*[0-9]*""".r
  final val FileMatch = """^filename (.*)""".r

  private[git] def parseBlame(blameLine1: String, blameLine2: String): Option[Blame] = {
    (blameLine1, blameLine2) match {
      case (BlameMatch(commit, originalLineNr, finalLineNr), FileMatch(filePath)) =>
        Some(Blame(commit, originalLineNr.toInt, finalLineNr.toInt, filePath))
      case _ =>
        None
    }
  }

  // Github sets a limit of 20000 lines for a diff.
  // https://docs.github.com/en/repositories/creating-and-managing-repositories/about-repositories#diff-limits
  // We are conservatively doubling that limit
  // TODO: We decided to disable the limit for now, so we set it to Int.MaxValue since we need to define
  //       a strategy to make it as little disruptive for users as possible
  private val DiffLineLimit = Int.MaxValue // 40000

  private def executeAndParseDiff(diffCommand: Seq[String], dir: java.io.File)(
      implicit logContext: ProjectLogContext
  ): CommitDiff = {
    var lineCount = 0

    val builder = GitCommandRunner
      .exec(diffCommand, Some(dir), Option(5.minutes))(new StringBuilder, (builder: StringBuilder, line: String) => {
        builder.append(line.replace("\r", ""))
        builder.append('\n')

        if (lineCount > DiffLineLimit) {
          throw new Exception(s"Exceeded diff lines limit of $DiffLineLimit lines")
        }

        lineCount += 1
        builder
      })

    val data = builder.result()
    if (data.startsWith("diff")) {
      val diffs = GitDiffParser.parse(data)
      CommitDiff(data, diffs)
    } else {
      // When the diff is empty
      CommitDiff("", Success(Seq.empty))
    }
  }
}
