package codacy.git.runners

import codacy.foundation.logging.Logger
import codacy.foundation.logging.context.LogContext

import java.io.InputStream
import java.nio.charset.CodingErrorAction
import scala.collection.mutable.ArrayBuffer
import scala.concurrent.{blocking, ExecutionContext, Future}
import scala.io.{BufferedSource, Codec}
import scala.sys.process._
import scala.util.{Failure, Try}

case class ProcessResult(result: Int, stdoutSeq: collection.Seq[String], stderrSeq: collection.Seq[String])

/**
  * Provides an execution context for managing the execution of external commands.
  *
  * @param command the sequence of command and its arguments to execute.
  * @constructor creates a new command execution context for the specified command.
  */
class CommandExecutionContext(command: Seq[String])(implicit logContext: LogContext) extends Logger {

  import CommandExecutionContext._

  /**
    * Buffer to capture standard output from the executed command.
    */
  private val stdoutBuffer = collection.mutable.ArrayBuffer[String]()

  /**
    * Buffer to capture standard error from the executed command.
    */
  private val stderrBuffer = collection.mutable.ArrayBuffer[String]()

  /**
    * I/O handler to interact with the process's input, output, and error streams.
    */
  private val processIO = new ProcessIO(writeInput => {
    writeInput.close()
  }, stdout => {
    writeToBufferAndClose(stdout, stdoutBuffer)
  }, stderr => {
    writeToBufferAndClose(stderr, stderrBuffer)
  })

  /**
    * The process corresponding to the executed command.
    */
  private val process = command.run(processIO)

  /**
    * Executes the command and returns a future representing the outcome.
    *
    * @param executionContext the execution context to run the future.
    * @return a Future containing the process result.
    */
  def executionFuture(implicit executionContext: ExecutionContext): Future[ProcessResult] = Future {
    blocking {
      val result = process.exitValue() // This will block until the process finishes
      ProcessResult(result, stdoutBuffer, stderrBuffer)
    }
  }

  /**
    * Cleans up resources and ensures the process is terminated.
    */
  def cleanup(): Unit = {
    // First log any error output
    if (stderrBuffer.nonEmpty) logger.warn(s"Command error output: ${stderrBuffer.mkString("\n")}")

    Try(process.destroy()) match {
      case Failure(e) => logger.warn("Error destroying the process", e)
      case _ =>
    }
  }

  /**
    * Writes lines from an input stream to the provided buffer.
    *
    * @param inputStream the input stream to read from.
    * @param buffer      the buffer to write lines to.
    */
  private def writeToBufferAndClose(inputStream: InputStream, buffer: ArrayBuffer[String]): Unit = {
    Try {
      new OnlyNewLineBufferedSource(inputStream).getLines().foreach(buffer += _)
      inputStream.close()
    } match {
      case Failure(e) =>
        logger.warn("Error processing the input stream", e)
        inputStream.close()
      case _ =>
    }
  }
}

object CommandExecutionContext {

  /**
    * Codec configuration for handling character decoding from input streams.
    */
  private val codec: Codec = {
    val codec0 = Codec.UTF8
    codec0.onMalformedInput(CodingErrorAction.IGNORE)
    codec0.onUnmappableCharacter(CodingErrorAction.IGNORE)
    codec0
  }

  /**
    * A source that provides lines from an input stream, splitting only at newline characters.
    *
    * @param inputStream the input stream to read from.
    */
  private class OnlyNewLineBufferedSource(inputStream: InputStream) extends BufferedSource(inputStream)(codec) {

    private class OnlyNewLineIterator extends LineIterator {
      override def isNewline(ch: Char): Boolean = ch == '\n'
    }

    override def getLines(): Iterator[String] = new OnlyNewLineIterator()
  }
}
