package codacy.git.runners

import codacy.foundation.logging.Logger
import codacy.foundation.logging.context.LogContext

import scala.concurrent.duration.Duration
import scala.concurrent.{ExecutionContext, Future, TimeoutException}
import scala.util.Failure

class CommandError(message: String) extends Throwable(message)

/**
  * Logic for executing shell commands.
  */
object CommandRunner extends Logger {

  /**
    * Runs a shell command with the specified timeout and returns the result as a Future.
    *
    * If the command execution times out or fails, the returned Future will fail with an appropriate exception.
    *
    * @param command the sequence of command and its arguments to execute.
    * @param timeout the maximum duration to wait for the command to complete.
    * @param executionContext the execution context in which to run the future.
    * @return a Future that, when completed, contains the sequence of output lines if the command was successful.
    *         Fails with a CommandError if the command reported an error.
    *         Fails with a TimeoutException if the command did not complete within the specified timeout.
    */
  def runCommand(
      command: Seq[String],
      timeout: Duration
  )(implicit executionContext: ExecutionContext, logContext: LogContext): Future[collection.Seq[String]] = {

    val commandLog = command.mkString("`", " ", "`")
    logger.info(s"Executing command: $commandLog with timeout: $timeout")

    val timeoutContext = new TimeoutContext(timeout)
    val commandExecutionContext = new CommandExecutionContext(command)

    Future
      .firstCompletedOf(Seq(commandExecutionContext.executionFuture, timeoutContext.timeoutFuture))
      .flatMap {
        case ProcessResult(result, stdoutSeq, _) if result == 0 => Future.successful(stdoutSeq)
        case ProcessResult(result, _, stderrSeq) =>
          val errorOutput = if (stderrSeq.isEmpty) "None" else stderrSeq.mkString("\n")
          logger.error(s"Command execution failed. Result: $result. Error output: $errorOutput")
          Future.failed(new CommandError(errorOutput))
      }
      .andThen {
        case Failure(_: TimeoutException) =>
          logger.error(s"Command execution timed out after $timeout. Command: $commandLog")
      }
      .andThen {
        case Failure(_) =>
          commandExecutionContext.cleanup()
      }
      .andThen {
        case _ =>
          timeoutContext.cleanup()
      }
  }
}
