package codacy.git.runners

import java.util.concurrent.{ScheduledExecutorService, ScheduledThreadPoolExecutor, TimeUnit}
import scala.concurrent.duration.Duration
import scala.concurrent.{Future, Promise, TimeoutException}

/**
  * Provides a context for managing timeouts for processes.
  *
  * @param timeout the duration to wait before triggering a timeout.
  * @constructor creates a new timeout context with the specified timeout duration.
  */
class TimeoutContext(timeout: Duration) {

  /**
    * A promise that will be completed with a failure if the timeout is reached.
    */
  private val timeoutPromise = Promise[ProcessResult]()

  /**
    * Scheduler service used to manage the execution delay for the timeout.
    */
  private val scheduler: ScheduledExecutorService = new ScheduledThreadPoolExecutor(1)

  // Initialize the scheduler to complete the `timeoutPromise` after the timeout duration.
  locally {
    scheduler.schedule(new Runnable {
      override def run(): Unit = {
        timeoutPromise.failure(new TimeoutException(s"Command execution timed out after $timeout"))
      }
    }, timeout.toMillis, TimeUnit.MILLISECONDS)
  }

  /**
    * Returns a Future that will be completed with the process result or will fail with a TimeoutException
    * if the timeout duration is exceeded.
    *
    * @return a Future of the process result.
    */
  def timeoutFuture: Future[ProcessResult] = timeoutPromise.future

  /**
    * Cleans up resources used by the TimeoutContext, specifically shutting down the scheduler.
    */
  def cleanup(): Unit = {
    scheduler.shutdown()
  }
}
