package codacy.utils

import codacy.exceptions.{DirectoryCreationException, DirectoryDeletionException}
import codacy.foundation.logging.context.{ContextLogging, ProjectLogContext}
import org.apache.commons.io.FileUtils

import java.io.File
import scala.util.{Failure, Success, Try}

object FileOperations extends ContextLogging {

  def createDirectory(dir: File)(implicit logContext: ProjectLogContext) = {
    Try {
      FileUtils.forceMkdir(dir)
    } match {
      case Success(_) =>
        logger.info(s"Created directory ${dir.getAbsolutePath}")
      case Failure(exception) =>
        logger.error(s"Failed to create directory ${dir.getAbsolutePath}")
        throw DirectoryCreationException(dir.getAbsolutePath, exception.getMessage)
    }
  }

  def deleteDirectory(projectDir: File)(implicit logContext: ProjectLogContext): Unit = {
    Try {
      logger.info(s"Deleting directory ${projectDir.getAbsolutePath}")
      FileUtils.deleteDirectory(projectDir)
    } match {
      case Success(_) =>
        logger.info(s"Deleted directory ${projectDir.getAbsolutePath}")
      case Failure(exception) =>
        logger.error(s"Failed to delete directory ${projectDir.getAbsolutePath}", exception)
        throw DirectoryDeletionException(projectDir.getAbsolutePath, exception.getMessage)
    }
  }
}
