package codacy.utils

import codacy.foundation.logging.context.ContextLogging

import java.io.File
import java.nio.channels.{FileChannel, OverlappingFileLockException}
import java.nio.file.{Paths, StandardOpenOption}
import java.util.concurrent.{CopyOnWriteArrayList, TimeoutException}

object FileSystemLocks {
  private val al = new CopyOnWriteArrayList[String]()

  sealed trait Lock
  private final case object Lock extends Lock
}

trait FileSystemLocks extends ContextLogging {
  import FileSystemLocks._

  private def fileNameToLockPath(f: File) = {
    val parentName = f.getParentFile()
    Paths.get(parentName.getAbsolutePath, f.getName + ".lock")
  }

  private def now() = System.currentTimeMillis()

  def lock[A](file: File, maxTime: Long = 1000 * 60 * 60)(method: (Lock) => A): A = {
    val startTime = now()
    val lockPath = fileNameToLockPath(file)

    file.getParentFile().mkdirs()

    val fileChannel = FileChannel.open(
      lockPath,
      StandardOpenOption.CREATE,
      StandardOpenOption.WRITE,
      StandardOpenOption.SYNC,
      StandardOpenOption.DELETE_ON_CLOSE
    )

    val fcName = lockPath.toAbsolutePath().toString()
    try {
      doLock(fileChannel, fcName)(method)(startTime + maxTime)
    } finally {
      al.remove(fcName)
      fileChannel.close()
    }
  }

  private def await(ms: Long) = Thread.sleep(ms)

  private def doLock[A](fc: FileChannel, fcName: String)(method: (Lock) => A)(endTime: Long): A = {
    var result: A = null.asInstanceOf[A]
    while (result == null && (now() < endTime)) {
      try {
        // File locks are held on behalf of the entire Java virtual machine.
        // They are not suitable for controlling access to a file by multiple threads within the same virtual machine.
        if (al.addIfAbsent(fcName)) {
          // this effectively does the `synchronized` across JVMs
          val lock = fc.lock(0, java.lang.Long.MAX_VALUE, false)
          if (lock.isValid() && lock.acquiredBy() == fc) {
            result = try { method(Lock) } finally { lock.release() }
          } else { await(100) } // retry
        } else { await(100) } // retry
      } catch {
        case _: OverlappingFileLockException => await(100) // retry
      }
    }
    if (now() >= endTime) {
      throw new TimeoutException("Timeout expired while trying to lock")
    } else {
      result
    }
  }

}
