## Set the SHELL option -o pipefail before RUN with a pipe in 

### Problematic code:

```dockerfile
RUN wget -O - https://some.site | wc -l > /number
```

### Correct code:

```dockerfile
SHELL ["/bin/bash", "-o", "pipefail", "-c"]
RUN wget -O - https://some.site | wc -l > /number
```

Or in case of busybox in an Alpine image:
```dockerfile
SHELL ["/bin/ash", "-eo", "pipefail", "-c"]
RUN wget -O - https://some.site | wc -l > /number
```

### Rationale:

> https://github.com/docker/docker.github.io/blob/master/develop/develop-images/dockerfile_best-practices.md#using-pipes

Some `RUN` commands depend on the ability to pipe the output of one command into another, using the pipe character (`|`), as in the following example:

```Dockerfile
RUN wget -O - https://some.site | wc -l > /number
```

Docker executes these commands using the `/bin/sh -c` interpreter, which
only evaluates the exit code of the last operation in the pipe to determine
success. In the example above this build step succeeds and produces a new
image so long as the `wc -l` command succeeds, even if the `wget` command
fails.

If you want the command to fail due to an error at any stage in the pipe,
prepend `set -o pipefail &&` to ensure that an unexpected error prevents
the build from inadvertently succeeding.

Since there are some shells that do not accept the `-o pipefail` option,
it is not enough to add `set -o pipefail` inside the `RUN` instruction. Therefore,
we recommend to always explicitly the `SHELL` before using pipes in `RUN`.