# Double quote to prevent globbing and word splitting.

## Problematic code:

```sh
echo $1
for i in $*; do :; done # this one and the next one also apply to expanding arrays.
for i in $@; do :; done
```

## Correct code:

```sh
echo "$1"
for i in "$@"; do :; done # or, 'for i; do'
```
## Rationale

The first code looks like "print the first argument". It's actually "Split the first argument by IFS (spaces, tabs and line feeds). Expand each of them as if it was a glob. Join all the resulting strings and filenames with spaces. Print the result."

The second one looks like "iterate through all arguments". It's actually "join all the arguments by the first character of IFS (space), split them by IFS and expand each of them as globs, and iterate on the resulting list". The third one skips the joining part.

Quoting variables prevents word splitting and glob expansion, and prevents the script from breaking when input contains spaces, line feeds, glob characters and such.

Strictly speaking, only expansions themselves need to be quoted, but for stylistic reasons, entire arguments with multiple variable and literal parts are often quoted as one:

```sh
$HOME/$dir/dist/bin/$file        # Unquoted (bad)
"$HOME"/"$dir"/dist/bin/"$file"  # Minimal quoting (good)
"$HOME/$dir/dist/bin/$file"      # Canonical quoting (good)
```

When quoting composite arguments, make sure to exclude globs and brace expansions, which lose their special meaning in double quotes: `"$HOME/$dir/src/*.c"` will not expand, but `"$HOME/$dir/src"/*.c` will.

Note that `$( )` starts a new context, and variables in it have to be quoted independently:

```sh
echo "This $variable is quoted $(but this $variable is not)"
echo "This $variable is quoted $(and now this "$variable" is too)"
```

### Exceptions
Sometimes you want to split on spaces, like when building a command line:

```sh
options="-j 5 -B"
make $options file
```

Just quoting this doesn't work. Instead, you should have used an array (bash, ksh, zsh):

```bash
options=(-j 5 -B) # ksh: set -A options -- -j 5 -B
make "${options[@]}" file
```

or a function (POSIX):

```sh
make_with_flags() { make -j 5 -B "$@"; }
make_with_flags file
```

To split on spaces but not perform glob expansion, Posix has a `set -f` to disable globbing.  You can disable word splitting by setting `IFS=''`.

Similarly, you might want an optional argument:

```sh
debug=""
[[ $1 == "--trace-commands" ]] && debug="-x"
bash $debug script
```

Quoting this doesn't work, since in the default case, `"$debug"` would expand to one empty argument while `$debug` would expand into zero arguments. In this case, you can use an array with zero or one elements as outlined above, or you can use an unquoted expansion with an alternate value:

```sh
debug=""
[[ $1 == "--trace-commands" ]] && debug="yes"
bash ${debug:+"-x"} script
```

This is better than an unquoted value because the alternative value can be properly quoted, e.g. `wget ${output:+ -o "$output"}`.

---

Here are two common cases where this warning seems unnecessary but may still be beneficial:

```sh
cmd <<< $var         # Requires quoting on Bash 3 (but not 4+)
: ${var=default}     # Should be quoted to avoid DoS when var='*/*/*/*/*/*'
```

---

As always, this warning can be [[ignored|ignore]] on a case-by-case basis.

This is especially relevant when BASH may not be available for the array work around.
For example, use in eval or in command options where script has total control of the variables...

```sh
FLAGS="-av -e 'ssh -x' --delete --delete-excluded"
...
# shellcheck disable=SC2086
eval rsync $FLAGS ~/dir remote_host:dir
```