# var was modified in a subshell. That change might be lost.

### Problematic code:

There are many ways of accidentally creating subshells, but a common one is piping to a loop:

```bash
n=0
printf "%s\n" {1..10} | while read i; do (( n+=i )); done
echo $n
```

### Correct code:

```bash
# Bash specific: process substitution. Also try shopts like lastpipe.
n=0
while read i; do (( n+=i )); done < <(printf "%s\n" {1..10})
echo $n
```

In `sh`, a temp file (better if fifo or fd) can be used instead of process substitution. And if it's acceptable to do it with waiting, try Here Documents.

### Rationale:

Variables set in subshells are not available outside the subshell. This is a wide topic, and better described on the [Wooledge Bash Wiki](http://mywiki.wooledge.org/BashFAQ/024).

Here are some constructs that cause subshells (shellcheck may not warn about all of them). In each case, you can replace `subshell1` by a command or function that sets a variable, e.g. simply `var=foo`, and the variable will appear to be unset after the command is run. Similarly, you can replace `regular` with `var=foo`, and it will be set afterwards:

Pipelines:

```sh
subshell1 | subshell2 | subshell3    # Bash, Dash, Ash
subshell1 | subshell2 | regular      # Ksh, Zsh
```

Command substitution:

```sh
regular "$(subshell1)" "`subshell2`"
```

Process substitution:

```sh
regular <(subshell1) >(subshell2)
```

Some forms of grouping:

```sh
( subshell )
{ regular; }
```

Backgrounding:

```sh
subshell1 &
subshell2 &
```

Anything executed by external processes:

```sh
find . -exec subshell1 {} \;
find . -print0 | xargs -0 subshell2
sudo subshell3
su -c subshell4
```

This applies not only to setting variables, but also setting shell options and changing directories.

### Exceptions

You can ignore this error if you don't care that the changes aren't reflected, because work on the value branches and shouldn't be recombined.

### Related resources:

* [BashFaq: I set variables in a loop that's in a pipeline. Why do they disappear after the loop terminates?](https://mywiki.wooledge.org/BashFAQ/024)
* [Bash Pitfalls: grep foo bar | while read -r; do ((count++)); done](https://mywiki.wooledge.org/BashPitfalls#pf8)
* [StackOverflow: A variable modified inside a while loop is not remembered](https://stackoverflow.com/questions/16854280/a-variable-modified-inside-a-while-loop-is-not-remembered)

[Source](https://github.com/koalaman/shellcheck/wiki/SC2031)

