## To expand via indirection, use name="foo$n"; echo "${!name}".

### Problematic code:

```sh
var_1="hello world"
n=1
echo "${var_$n}"
```

### Correct code:

Bash/ksh:

```sh
# Use arrays instead of dynamic names
declare -a var
var[1]="hello world"
n=1
echo "${var[n]}"
```

or

```sh
# Expand variable names dynamically
var_1="hello world"
n=1
name="var_$n"
echo "${!name}"
```

POSIX sh:

```sh
# Expand dynamically with eval
var_1="hello world"
n=1
eval "tmp=\$var_$n"
echo "${tmp}"
```

### Rationale:

You can expand a variable `var_1` with `${var_1}`, but you can not generate the string `var_1` with an embedded expansion, like `${var_$n}`.

Instead, if at all possible, you should use an array. Bash and ksh support both numerical and associative arrays, and an example is shown above.

If you can't use arrays, you can indirectly reference variables by creating a temporary variable with its name, e.g. `myvar="var_$n"` and then expanding it indirectly with `${!myvar}`. This will give the contents of the variable `var_1`.

If using POSIX sh, where neither arrays nor `${!var}` is available, `eval` can be used. You must be careful in sanitizing the data used to construct the variable name to avoid arbitrary code execution.

### Exceptions:

None

### Related resources:

* [BashFaq: How can I use variable variables (indirect variables, pointers, references) or associative arrays?](https://mywiki.wooledge.org/BashFAQ/006)
* [StackOverflow: Dynamic variable names in Bash](https://stackoverflow.com/questions/16553089/dynamic-variable-names-in-bash)
[Source](https://github.com/koalaman/shellcheck/wiki/SC2082)

