
Controlling permissions is security-sensitive. It has led in the past to the following vulnerabilities:

- [CVE-2018-12999](http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2018-12999)
- [CVE-2018-10285](http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2018-10285)
- [CVE-2017-7455](http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2017-7455)


Attackers can only damage what they have access to. Thus limiting their access is a good way to prevent them from wreaking havoc, but it has to be done properly.

This rule flags code that controls the access to resources and actions or configures this access. The goal is to guide security code reviews.

## Ask Yourself Whether

- at least one accessed action or resource is security-sensitive.
- there is no access control in place or it does not cover all sensitive actions and resources.
- users have permissions they don't need.
- the access control is based on a user input or on some other unsafe data.
- permissions are difficult to remove or take a long time to be updated.


You are at risk if you answered yes to the first question and any of the following ones.

## Recommended Secure Coding Practices

The first step is to restrict all sensitive actions to authenticated users.

Each user should have the lowest privileges possible. The access control granularity should match the sensitivity of each resource or action. The more sensitive it is, the less people should have access to it.

Do not base the access control on a user input or on a value which might have been tampered with. For example, the developer should not read a user's permissions from an HTTP cookie as it can be modified client-side.

Check that the access to each action and resource is properly restricted.

Enable administrators to swiftly remove permissions when necessary. This enables them to reduce the time an attacker can have access to your systems when a breach occurs.

Log and monitor refused access requests as they can reveal an attack.

## Sensitive Code Example


    Imports System.Threading
    Imports System.Security.Permissions
    Imports System.Security.Principal
    Imports System.IdentityModel.Tokens
    
    Class SecurityPrincipalDemo
        Class MyIdentity
            Implements IIdentity ' Sensitive, custom IIdentity implementations should be reviewed
        End Class
    
        Class MyPrincipal
            Implements IPrincipal ' Sensitive, custom IPrincipal implementations should be reviewed
        End Class
    
        &lt;System.Security.Permissions.PrincipalPermission(SecurityAction.Demand, Role:="Administrators")&gt; ' Sensitive. The access restrictions enforced by this attribute should be reviewed.
        Private Shared Sub CheckAdministrator()
            Dim MyIdentity As WindowsIdentity = WindowsIdentity.GetCurrent() ' Sensitive
    
            HttpContext.User = ... ' Sensitive: review all reference (set and get) to System.Web HttpContext.User
    
            Dim domain As AppDomain = AppDomain.CurrentDomain
            domain.SetPrincipalPolicy(PrincipalPolicy.WindowsPrincipal) ' Sensitive
    
            Dim identity As MyIdentity = New MyIdentity() ' Sensitive
            Dim MyPrincipal As MyPrincipal = New MyPrincipal(MyIdentity) ' Sensitive
            Thread.CurrentPrincipal = MyPrincipal ' Sensitive
            domain.SetThreadPrincipal(MyPrincipal) ' Sensitive
    
            Dim principalPerm As PrincipalPermission = New PrincipalPermission(Nothing, "Administrators")  ' Sensitive
            principalPerm.Demand()
    
            Dim handler As SecurityTokenHandler = ...
            Dim identities As ReadOnlyCollection(Of ClaimsIdentity) = handler.ValidateToken()  ' Sensitive, this creates identity
        End Sub
    
        ' Sensitive: review how this function uses the identity and principal.
        Private Sub modifyPrincipal(ByVal identity As MyIdentity, ByVal principal As MyPrincipal)
        End Sub
    End Class


## See

- [OWASP Top 10 2017 Category A5](https://www.owasp.org/index.php/Top_10-2017_A5-Broken_Access_Control) - Boken Access Control
- [SANS Top 25](https://www.sans.org/top25-software-errors/#cat3) - Porous Defenses

