## Declare and assign separately to avoid masking return values.

### Problematic code in the case of `export`:

```sh
export foo="$(mycmd)"
```

### Correct code:

```sh
foo="$(mycmd)"
export foo
```

### Rationale:

In the original code, the return value of `mycmd` is ignored, and `export` will instead always return true. This may prevent conditionals, `set -e` and traps from working correctly.

When first marked for export and assigned separately, the return value of the assignment will be that of `mycmd`. This avoids the problem.

### Exceptions:

If you intend to ignore the return value of an assignment, you can either ignore this warning or use

```sh
foo=$(mycmd) || true
export foo
```

Shellcheck does not warn about `export foo=bar` because `bar` is a literal and not a command substitution with an independent return value. It also does not warn about `local -r foo=$(cmd)`, where declaration and assignment must be in the same command.

### Problematic code in the case of `local`:

```sh
local foo="$(mycmd)"
```

### Correct code:

```sh
local foo
foo=$(mycmd)
```

### Rationale

The exit status of the command is overridden by the exit status of the creation of the local variable. For example:

```bash
$ f() { local foo=$(false) || echo foo; }; f
$ f() { local foo; foo=$(false) || echo foo; }; f
foo
```

### Problematic code in the case of `readonly`:

```sh
readonly foo="$(mycmd)"
```

### Correct code:

```sh
foo="$(mycmd)"
readonly foo
```

[Source](https://github.com/koalaman/shellcheck/wiki/SC2155)

