
An `IDisposable` object should be disposed (there are some rare exceptions where not disposing is fine, most notably `Task`). If a class has an `IDisposable` field, there can be two situations:

- The class observes a field that it under the responsibility of another class.

- The class owns the field, and is therefore responsible for calling `Dispose` on it.

In the second case, the safest way for the class to ensure `Dispose` is called is to call it in its own `Dispose` function, and therefore to be itself `IDisposable`. A class is considered to own an `IDisposable` field resource if it created the object referenced by the field.

## Noncompliant Code Example


    public class ResourceHolder   // Noncompliant; doesn't implement IDisposable
    {
      private FileStream fs;  // This member is never Disposed
      public void OpenResource(string path)
      {
        this.fs = new FileStream(path, FileMode.Open); // I create the FileStream, I'm owning it
      }
      public void CloseResource()
      {
        this.fs.Close();
      }
    }


## Compliant Solution


    public class ResourceHolder : IDisposable
    {
    &nbsp;&nbsp;private FileStream fs;
    &nbsp;&nbsp;public void OpenResource(string path)
    &nbsp;&nbsp;{
    &nbsp;&nbsp;&nbsp;&nbsp;this.fs = new FileStream(path, FileMode.Open); // I create the FileStream, I'm owning it
    &nbsp;&nbsp;}
    &nbsp;&nbsp;public void CloseResource()
    &nbsp;&nbsp;{
    &nbsp;&nbsp;&nbsp;&nbsp;this.fs.Close();
    &nbsp;&nbsp;}
    
    &nbsp;&nbsp;public void Dispose()
    &nbsp;&nbsp;{
    &nbsp;&nbsp;&nbsp;&nbsp;this.fs.Dispose();
    &nbsp;&nbsp;}
    }


## See

- [MITRE, CWE-459](http://cwe.mitre.org/data/definitions/459.html) - Incomplete Cleanup

