
Most of cryptographic systems require a sufficient key size to be robust against brute-force attacks.

[NIST recommendations](https://nvlpubs.nist.gov/nistpubs/SpecialPublications/NIST.SP.800-131Ar2.pdf) will be checked for these use-cases:

**Digital Signature Generation** and **Verification:**

- p ≥ 2048 AND q ≥ 224 for DSA (`p` is key length and `q` the modulus length)
- n ≥ 2048 for RSA (`n` is the key length)


**Key Agreement**:

- p ≥ 2048 AND q ≥ 224 for DH and MQV
- n ≥ 224 for ECDH and ECMQV (Examples: `secp192r1` is a non-compliant curve (`n` &lt; 224) but `secp224k1` is<br>  compliant (`n` &gt;= 224))


**Encryption and Decryption**:

- AES-128, 192, 256


This rule will not raise issues for ciphers that are considered weak (no matter the key size) like `DES`, `Blowfish`.

## Noncompliant Code Example


    using System;
    using System.Security.Cryptography;
    
    namespace MyLibrary
    {
        public class MyCryptoClass
        {
            static void Main()
            {
                var dsa1 = new DSACryptoServiceProvider(); // Noncompliant - default key size is 1024
                dsa1.KeySize = 2048; // Noncompliant - the setter does not update the underlying key size for the DSACryptoServiceProvider class
    
                var dsa2 = new DSACryptoServiceProvider(2048); // Noncompliant - cannot create DSACryptoServiceProvider with a key size bigger than 1024
    
                var rsa1 = new RSACryptoServiceProvider(); // Noncompliant - default key size is 1024
                rsa1.KeySize = 2048; // Noncompliant - the setter does not update the underlying key size for the RSACryptoServiceProvider class
    
                var rsa2 = new RSACng(1024); // Noncompliant
    
                // ...
            }
        }
    }


KeySize property of DSACryptoServiceProvider and RSACryptoServiceProvider does not change the value of underlying KeySize for the algorithm. Property setter is ignored without error and KeySize can be changed only by using constructor overload. See:

- [DSACryptoServiceProvider.KeySize<br>  Property](https://docs.microsoft.com/en-us/dotnet/api/system.security.cryptography.dsacryptoserviceprovider.keysize)
- [RSACryptoServiceProvider.KeySize<br>  Property](https://docs.microsoft.com/en-us/dotnet/api/system.security.cryptography.rsacryptoserviceprovider.keysize)


## Compliant Solution


    using System;
    using System.Security.Cryptography;
    
    namespace MyLibrary
    {
        public class MyCryptoClass
        {
            static void Main()
            {
                var dsa1 = new DSACng(); // Compliant - default key size is 2048
                var dsa2 = new DSACng(2048); // Compliant
                var rsa1 = new RSACryptoServiceProvider(2048); // Compliant
                var rsa2 = new RSACng(); // Compliant - default key size is 2048
    
                // ...
            }
        }
    }


## See

- [OWASP Top 10 2017 Category A3](https://www.owasp.org/index.php/Top_10-2017_A3-Sensitive_Data_Exposure) - Sensitive Data Exposure<br>
- [OWASP Top 10 2017 Category A9](https://www.owasp.org/index.php/Top_10-2017_A6-Security_Misconfiguration) - Security<br>  Misconfiguration
- [NIST 800-131A](https://nvlpubs.nist.gov/nistpubs/SpecialPublications/NIST.SP.800-131Ar1.pdf) - Recommendation for Transitioning the<br>  Use of Cryptographic Algorithms and Key Lengths
- [MITRE, CWE-326](http://cwe.mitre.org/data/definitions/326.html) - Inadequate Encryption Strength

