# [Potential Path Traversal (file read)](https://find-sec-bugs.github.io/bugs.htm#PATH_TRAVERSAL_IN)

A file is opened to read its content. The filename comes from an **input** parameter.
If an unfiltered parameter is passed to this file API, files from an arbitrary filesystem location could be read.

This rule identifies **potential** path traversal vulnerabilities. In many cases, the constructed file path cannot be controlled
by the user. If that is the case, the reported instance is a false positive.

    **Vulnerable Code:**  

<pre>@GET
@Path("/images/{image}")
@Produces("images/*")
public Response getImage(@javax.ws.rs.PathParam("image") String image) {
    File file = new File("resources/images/", image); //Weak point

    if (!file.exists()) {
        return Response.status(Status.NOT_FOUND).build();
    }

    return Response.ok().entity(new FileInputStream(file)).build();
}</pre>

    **Solution:**  

<pre>import org.apache.commons.io.FilenameUtils;

@GET
@Path("/images/{image}")
@Produces("images/*")
public Response getImage(@javax.ws.rs.PathParam("image") String image) {
    File file = new File("resources/images/", FilenameUtils.getName(image)); //Fix

    if (!file.exists()) {
        return Response.status(Status.NOT_FOUND).build();
    }

    return Response.ok().entity(new FileInputStream(file)).build();
}</pre>

**References**  

[WASC: Path Traversal](http://projects.webappsec.org/w/page/13246952/Path%20Traversal)  

[OWASP: Path Traversal](https://www.owasp.org/index.php/Path_Traversal)  

[CAPEC-126: Path Traversal](https://capec.mitre.org/data/definitions/126.html)  

[CWE-22: Improper Limitation of a Pathname to a Restricted Directory ('Path Traversal')](https://cwe.mitre.org/data/definitions/22.html)