package codacy.patterns

import codacy.base.Pattern

import scala.meta._
import scala.util.Try

case object Custom_Scala_HttpOnlyCookie extends Pattern{

  override def apply(tree: Tree) = {
    val finders = Seq(
      new SprayFinder(tree),
      new FinagleFinder(tree),
      new PlayFinder(tree)
    ).filter(_.hasFrameworkHints)
    val places = finders.flatMap(_.findCookies)
    places.map { case item => Result(message(item), item)}
  }

  private[this] def message(tree: Tree) = Message("Prohibit creating cookies without HttpOnly set.")

  private[this] trait FrameworkFinder {
    def hasFrameworkHints: Boolean
    def findCookies: Seq[Tree]
  }

  private[this] class FinagleFinder(val tree: Tree) extends FrameworkFinder {

    private[this] val constructorName = "Cookie"
    private[this] val fieldName = "httpOnly"

    def hasFrameworkHints: Boolean = {
      tree.collect{
        case importer"com.twitter.finagle.http._" => true
        case importer"com.twitter.finagle.Cookie" => true
      }.exists(identity)
    }

    private[this] def isExpectedConstructor(expr: Tree): Boolean = {
      expr.collect {
        case init"${ctorname: Type}(..${ctornameexprssnel: Seq[Term]})" if ctorname.toString == constructorName => ctorname
      }.nonEmpty
    }

    def findCookies: Seq[Tree] = {
      tree.collect {
        case q"..${mods:Seq[Mod]} val ..${patsnel:Seq[Pat]}: $tpeopt = ${expr: Tree}"
          if isExpectedConstructor(expr) && !definesSecureCookie(expr, patsnel) => expr
      }.toSeq
    }

    private[this] def determineScope(node: Tree): Option[Tree] = {
      node match {
        case t@q"..$mods def $name[..$tparams](...$paramss): $tpeopt = $expr" => Some(t)
        case t@q"..$mods class $name (...$params) extends $template" => Some(t)
        case source"..$stats" => Option.empty
        case item => item.parent.flatMap(determineScope)
      }
    }

    private[this] def literalBooleanValue(lit: Lit): Boolean = {
      (lit.value match {
        case t: Boolean => Some(t)
        case _ => None
      }).getOrElse(true)
    }

    private[this] def insituNames(tree: Tree): Seq[(Term, Term.Name, Tree)] = {
      tree.collect {
        case q"${ref: Term}.${attr: Term.Name} = ${expr: Tree}" if attr.toString == fieldName =>
          (ref, attr, expr)
      }
    }

    private[this] def definesSecureCookie(expr: Tree, fields: Seq[Pat]) = {
      val names: Seq[Name] = fields.flatMap(_.collect { case n: Term.Name => n })
      val scope = determineScope(expr)
      scope.fold(true) { case tree =>
        val foundNames = insituNames(tree).filter{
          case (ref, _, _) => names.exists(_.toString == ref.toString)}
        if (foundNames.isEmpty) {
          false
        } else {
          foundNames.map { case (ref, attr, expr) =>
            val literalValues = expr.collect{
              case q"${lit: Lit}" => literalBooleanValue(lit)
            }
            if (literalValues.nonEmpty) { literalValues.forall(identity) } else { true }
          }.forall(identity)
        }
      }
    }
  }

  private[this] class SprayFinder(val tree: Tree) extends FrameworkFinder {

    private[this] val constructorName = "HttpCookie"
    private[this] val flagName = "httpOnly"
    // The seventh argument is the httpOnly parameter.
    private[this] val position = 7

    def hasFrameworkHints: Boolean = {
      tree.collect{
        case importer"spray.http._" => true
      }.exists(identity)
    }

    def findCookies: Seq[Tree] = {
      tree.collect {
        case init"${ctorname: Type}(..${ctornameexprssnel: Seq[Term]})"
          if ctorname.toString == constructorName && definesInsecureCookie(ctornameexprssnel) => ctorname
        case q"$expr(..$exprssnel)"
          if expr.toString == constructorName && definesInsecureCookie(exprssnel) => expr
      }.toSeq
    }

    private[this] def definesInsecureCookie(args: Seq[Term]): Boolean = {
      val httpOnlyKeywordArgs = args.filter(isHttpOnlyKeyword)
      if (httpOnlyKeywordArgs.nonEmpty) {
        val values = httpOnlyKeywordArgs.flatMap(httpOnlyValue)
        val isExplicitInsecure = values.contains(false)
        isExplicitInsecure
      } else {
        val valueOfArgument = args.lift(position)
        valueOfArgument match {
          case Some(value) =>
            value.collect {
              case q"${lit: Lit}" if !asBoolean(lit.value).getOrElse(true) => true
            }.nonEmpty
          case _ => true
        }
      }
    }

    private[this] def isHttpOnlyKeyword(arg: Term): Boolean = {
      arg.collect {
        case q"${name: Name}" if name.toString == flagName => true
      }.nonEmpty
    }

    private[this] def asBoolean(value: Any): Option[Boolean] = {
      value match {
        case t: Boolean => Some(t)
        case _ => None
      }
    }

    private[this] def httpOnlyValue(arg: Term): Option[Boolean] = {
      // TODO: how can I do compile time checking for quasiquotes?
      val result: Seq[Boolean] = arg.collect {
        case q"${lit: Lit}" => asBoolean(lit.value)
      }.flatten
      result.headOption
    }
  }

  private[this] class PlayFinder(tree:Tree) extends FrameworkFinder{

    override def hasFrameworkHints: Boolean = tree.collect{
      case importer"play.api.mvc._" => true
      case importer"play.api.mvc.Cookie" => true
    }.exists(identity)

    override def findCookies: Seq[Tree] = {
      tree.collect{
        case t@init"$tpe(...$exprss)" if isCookieExpr(tpe) && hasInsecureArg(exprss.flatten) => t
        case t@q"$expr(..${aexprssnel:Seq[Term]})" if isCookieExpr(expr) && hasInsecureArg(aexprssnel) => t
      }
    }

    private[this] def isCookieExpr(expr:Tree):Boolean = {
      expr match{
        case q"Cookie" => true
        case init"Cookie" => true
        case q"$_.Cookie" => true
        case _ if expr.toString endsWith "Cookie" => true
        case _ => false
      }
    }

    private[this] def hasInsecureArg(args:Seq[Term]):Boolean = {
      val namedArgValue = args.collectFirst{ case t:Term.Assign if t.lhs.toString() == "httpOnly" => t }
      //no named args?
      lazy val setsUnnamedOrDefaultArg = namedArgValue.isEmpty && (args.toList match{
        case _ :: _ :: _ :: _ :: _ :: _ :: httpArg :: Nil =>
          httpArg match{
            case Lit(false) => true
            case other => false
          }
        case listWithDefault => false
      })

      lazy val setsNamedArg = namedArgValue.collect{ case Term.Assign(_, Lit(false)) => true }.getOrElse(false)
      setsNamedArg || setsUnnamedOrDefaultArg
    }
  }

}