## In POSIX sh, *something* is undefined.
You have declared that your script works with `/bin/sh`, but you are using features that have undefined behavior according to the POSIX specification. 

It may currently work for you, but it can or will fail on other OS, the same OS with different configurations, from different contexts (like initramfs/chroot), or in different versions of the same OS, including future updates to your current system.

Either declare that your script requires a specific shell like `#!/bin/bash` or `#!/bin/dash`, or rewrite the script in a portable way.

For help with rewrites, the Ubuntu wiki has [a list of portability issues](https://wiki.ubuntu.com/DashAsBinSh) that broke people's `#!/bin/sh` scripts when Ubuntu switched from Bash to Dash. See also [Bashism](http://mywiki.wooledge.org/Bashism) on wooledge's wiki. ShellCheck may not warn about all these issues.

### `$'c-style-escapes'`

bash, ksh:

```Bash
a=$' \t\n'
```

POSIX:

```sh
a="$(printf '%b_' ' \t\n')"; a="${a%_}" # protect trailing \n
```

Want some good news? See http://austingroupbugs.net/view.php?id=249#c590.

### `$"msgid"`

Bash:

```Bash
echo $"foo $(bar) baz"
```

POSIX:

```sh
. gettext.sh # GNU Gettext sh library
# ...
barout=$(bar)
eval_gettext 'foo $barout baz' # See GNU Gettext doc for more info.
```

Or you can change them to normal double quotes so you go without `gettext`.

### Arithmetic `for` loops

Bash:

```Bash
for ((init; test; next)); do foo; done
```

POSIX:

```sh
: $((init))
while [ $((test)) -ne 0 ]; do foo; : $((next)); done
```

### Arithmetic exponentiation

Bash:

```Bash
printf "%s\n" "$(( 2**63 ))"
```

POSIX:

The POSIX standard does not allow for exponents.  However, you can replicate them completely built-in using a POSIX compatible function.  As an example, the `pow` function from [here](http://unix.stackexchange.com/a/7925).

```sh
pow () {
    set "$1" "$2" 1
    while [ "$2" -gt 0 ]; do
      set "$1" $(($2-1)) $(($1*$3))
    done
    # %d = signed decimal, %u = unsigned decimal
    # Either should overflow to 0
    printf "%d\n" "$3"
}
```

To compare:

```sh
$ echo "$(( 2**62 ))"
4611686018427387904
$ pow 2 62
4611686018427387904
```

Alternatively, if you don't mind using an external program, you can use `bc`.  Be aware though: `bash` and other programs may abide by a certain maximum integer that `bc` does not (for `bash` that's: 64-bit signed long int, failing back to 32-bit signed long int).  

Example:

```Bash
# Note the overflow that gives a negative number
$ echo "$(( 2**63 ))"
-9223372036854775808

# No such problem
$ echo 2^63 | bc
9223372036854775808

# 'bc' just keeps on going
$ echo 2^1280 | bc
20815864389328798163850480654728171077230524494533409610638224700807\
21611934672059602447888346464836968484322790856201558276713249664692\
98162798132113546415258482590187784406915463666993231671009459188410\
95379622423387354295096957733925002768876520583464697770622321657076\
83317005651120933244966378183760369413644440628104205339687097746591\
6057756101739472373801429441421111406337458176
```

### standalone `((..))`

Bash:

```Bash
((a=c+d))
((d)) && echo d is true.
```

POSIX:

```sh
: $((a=c+d)) # discard the output of the arith expn with `:` command
[ $((d)) -ne 0 ] && echo d is true. # manually check non-zero => true
```

### `select` loops

It takes extra care over terminal columns to make select loop look like bash's, which generates a list with multiple items on one line, or like `ls`.

It is, however, still possible to make a naive translation for `select foo in bar baz; do eat; done`:

```sh
while
  _i=0 _foo= foo=
  for _name in bar baz; do echo "$((_i+=1))) $_name"; done
  printf '$# '; read _foo
do
  case _foo in 1) foo=bar;; 2) foo=baz;; *) continue;; esac
  eat
done
```

### Here-strings

Bash, ksh:

```Bash
grep aaa <<< "$g"
```

POSIX:

```sh
# not exactly the same -- <<< adds a trailing \n if $g doesn't end with \n
printf '%s' "$g" | grep aaa
```

### echo flags

See https://unix.stackexchange.com/tags/echo/info.

### `${var/pat/replacement}`

Bash:

```Bash
echo "${TERM/%-256*}"
```

POSIX:

```sh
echo "$TERM" | sed -e 's/-256.*$//g'
# Special case for this since we are matching the end:
echo "${TERM%-256*}"
```

### `printf %q`

Bash:

```Bash
printf '%q ' "$@"
```

POSIX:

```sh
# TODO: Interpret it back to printf escapes for hard-to-copy chars like \t?
# See also: http://git.savannah.gnu.org/cgit/libtool.git/tree/gl/build-aux/funclib.sh?id=c60e054#n1029
reuse_quote()(
  for i; do
    __i_quote=$(printf '%s\n' "$i" | sed -e "s/'/'\\\\''/g"; echo x)
    printf "'%s'" "${__i_quote%x}"
  done
)
reuse_quote "$@"
```

## Exception

Depends on what your expected POSIX shell providers would use.
[Source](https://github.com/koalaman/shellcheck/wiki/SC2039)

