## Piping to 'rm', a command that doesn't read stdin. Wrong command or missing xargs?

### Problematic code:

```sh
ls | echo                      # Want to print result
cat files | rm                 # Want to delete items from a file
find . -type f | cp dir        # Want to process 'find' output
rm file | true                 # Want to ignore errors
```

### Correct code:

```sh
ls
cat files | while IFS= read -r file; do rm -- "$file"; done
find . -type f -exec cp {} dir \;
rm file || true
```

### Rationale:

You are piping to one of several commands that don't read from stdin.

This may happen when:

* Confusing one command for another, e.g. using `echo` where `cat` was intended.
* Incorrectly refactoring, leaving a `|` on the previous line.
* Missing `xargs`, because stdin should be passed as positional parameters instead (use `xargs -0` if at all possible).
* Intending to use `||` instead of `|`

Check your logic, and rewrite the command so data is passed correctly.

### Exceptions:

If you've overridden a command to return output, you can either rename it to make this obvious, or [[ignore]] this message.
[Source](https://github.com/koalaman/shellcheck/wiki/SC2216)

