## Detected permissive file permissions

### Problematic:

```python
with open('testfile.txt', 'w') as fout:
    fout.write("secrets!")
```


### Correct:

```python
import os

flags = os.O_WRONLY | os.O_CREAT | os.EXLC
with os.fdopen(os.open('testfile.txt', flags, 0o600), 'w') as fout:
    fout.write("secrets!")
```

### Rationale:

Files should be created with restrictive file permissions to prevent vulnerabilities such as information disclosure and code execution.
When writing to a file on a *NIX operating system using Python, the file output will be written using the umask that is set for the application. Depending on your operating system configuration this could be too permissive when writing sensitive data to file.
It is particularly important to note which other users are part of a group that you grant access to. The best practice is that if group access is not needed, do not grant it. This is the principle of least privilege.

[Reference](https://docs.openstack.org/bandit/latest/plugins/b103_set_bad_file_permissions.html)

See also:

* https://security.openstack.org/guidelines/dg_apply-restrictive-file-permissions.html
* https://en.wikipedia.org/wiki/File_system_permissions

