package codacy.patterns

import codacy.base.Pattern

import scala.meta._
import scala.util.Try

case class Sink(val methodName: String, val position: Int, val hints: Seq[String])

case class CallSite(ctorname: Tree, val args: Seq[Term], val sink: Sink)

trait FrameworkTrail {
  val sinks: Seq[Sink]
}

class SprayTrail(tree: Tree) extends FrameworkTrail {
  val sinks = Seq(
    Sink("BasicHttpCredentials", 1, hints = Seq("spray.http._"))
  )
}

class FinagleTrail(tree: Tree) extends FrameworkTrail {
  val sinks = Seq(
    Sink("Mysql.client.withCredentials", 1,
      hints = Seq("com.twitter.finagle.exp.Mysql", "com.twitter.finagle.exp._")),
    Sink("UsernamePassAuthenticationSetting", 1,
      hints = Seq("com.twitter.finagle.socks._", "com.twitter.finagle.socks.UsernamePassAuthenticationSetting")),
    Sink("ProxyCredentials", 1,
      hints = Seq("com.twitter.finagle.http._", "com.twitter.finagle.http.ProxyCredentials")),
    Sink("Credentials", 1,
      hints = Seq("com.twitter.finagle.client._", "com.twitter.finagle.client.Transporter"))
  )
}

class JavaTrail(tree: Tree) extends FrameworkTrail {
  val sinks = Seq(
    Sink("DriverManager.getConnection", 1,
      hints = Seq("java.sql._", "java.sql.DriverManager")),
    Sink("KeyStore.PasswordProtection", 0,
      hints = Seq("java.security._", "java.security.KeyStore")),
    Sink("PasswordAuthentication", 1,
      hints = Seq("java.net._", "java.net.PasswordAuthentication")),
    Sink("setPassword", 0,
      hints = Seq("javax.security.auth.callback._", "javax.security.auth.callback.PasswordCallback")),
    // This might introduce false positives
    Sink("load", 0,
      hints = Seq("java.security.KeyStore")),

    Sink("KerberosKey", 1,
      hints = Seq("javax.security.auth.kerberos._", "javax.security.auth.kerberos.KerberosKey"))
  )
}

class Finder(tree: Tree, trail: FrameworkTrail) {

  private[this] def determineScope(node: Tree): Option[Tree] = {
    node match {
      case t@q"..$mods def $name[..$tparams](...$paramss): $tpeopt = $expr" => Some(t)
      case t@q"..$mods class $name (...$params) extends $template" => Some(t)
      case source"..$stats" => Option.empty
      case item => item.parent.flatMap(determineScope)
    }
  }

  // Support shadowing of local variables. This should avoid some
  // edge cases where a shadowing name has the same name as the
  // hardcoded value.
  private[this] def hasConstructorChild(stats: Tree, child: Tree) = {
    stats.collect{
      case t@init"${ctorname: Term}" if ctorname == child => true
    }.nonEmpty
  }

  private[this] def sameShadowedParameter(param: Term.Param, name: Term.Name): Boolean = {
    param.collect{
      case param"..$mods ${paramname: Name}: $atpeopt = $exprop" if paramname.toString() == name.toString() => true
    }.nonEmpty
  }

  private[this] def isStringParam(param: Name, atpeopt: Option[Type]): Boolean = {
    atpeopt.getOrElse("").toString == "String"
  }

  private[this] def findShadowingInBlock(stats: Tree, scope: Tree, name: Term.Name, callSite: CallSite) = {
    stats.collect{
      case q"(..${params: Seq[Term.Param]}) => $expr"
        if hasConstructorChild(stats, callSite.ctorname) &&
          params.exists(param => sameShadowedParameter(param, name)) => params
      case p"case ${pat: Pat} if $expropt => $expr"
        if hasConstructorChild(stats, callSite.ctorname) &&
          pat.collect{
            case p"${patname: Term.Name}" if patname.toString == name.toString => name
          }.nonEmpty => Seq(pat)
    }.flatten.nonEmpty || scope.collect{
      case param"..$mods ${param: Name}: $atpeopt = $exprop"
        if param.value == name.toString &&
          isStringParam(param, atpeopt) &&
          determineScope(param).exists{ case paramScope => paramScope == scope} => true
    }.nonEmpty
  }

  private[this] def isShadowed(name: Term.Name, callSite: CallSite, scope: Tree): Boolean = {
    tree.collect{
      case t@q"{ ..$stats }" => findShadowingInBlock(t, scope, name, callSite)
      case t@q"{ ..case $casesnel }" => findShadowingInBlock(t, scope, name, callSite)
    }.toSeq.exists(identity)
  }

  private[this] def hardCodedScopeValues(tree: Tree): List[(scala.meta.Tree, Seq[scala.meta.Term.Name])] = {
    tree.collect {
      //valDefs
      case t@q"..${mods: Seq[Mod]} val ..${patsnel: Seq[Pat]}: $tpeopt = ${expr: Lit}" if isCharOrStringLiteral(expr.value) =>
        //first parent should be the stats List[Tree], 2nd the class itself
        flattenedParent(t, patsnel)
      case t@q"..${mods: Seq[Mod]} val ..${patsnel: Seq[Pat]}: $tpeopt = ${expr: Term}" if isHardCodedArray(expr) =>
        flattenedParent(t, patsnel)
    }.flatten
  }

  private[this] val scopedValues = hardCodedScopeValues(tree)

  private[this] val pattern = "(?i)(pass|pwd|psw|secret|key|cipher|crypt|des|aes|mac|private|sign|cert).*".r

  private[this] def isSuspiciousName(name: Term.Name, callSite: CallSite): Boolean = {
    pattern.findFirstMatchIn(name.toString) match {
      case Some(matched) =>
        val isHardCodedSomewhere = scopedValues.exists{
          case (tree, names) =>
            names.map(_.toString).contains(name.toString)
        }
        val scope = determineScope(name)
        val shadowed = scope.fold(false){ case scope => isShadowed(name, callSite, scope)}
        isHardCodedSomewhere && !shadowed
      case _ => false
    }
  }

  private[this] def flattenedParent(t: Tree, patsnel: Seq[Pat]) = {
    t.parent.flatMap(_.parent).map { case classDef =>
      (classDef, patsnel.flatMap(_.collect { case p"${name: Term.Name}" => name }))
    }
  }

  private[this] def callSites(): Seq[(Tree, Seq[Term])] = {
    tree.collect {
      case q"new $expr(...$exprss)" => (expr, exprss.flatten)
      case q"new $expr(...$exprss) with ..$inits { $self => ..$stats }" => (expr, exprss.flatten)
      case q"$expr(...$exprss)" => (expr, exprss.flatten)
    }
  }

  private[this] def sameSinkConstructor(sink: Sink, ctorname: Tree): Boolean = {
    val subref = ctorname.collect{
      case init"${ctorname: Type}(..$_)" =>
        Try(ctorname.toString.split('.').last).getOrElse(ctorname)
      case q"$ref.$ctorname" =>
        ctorname.toString
    }.headOption.getOrElse("")

    sink.methodName == ctorname.toString() || subref == sink.methodName
  }

  private[this] def matchedCallSites(): Seq[CallSite] = {
    val currentCallSites = callSites()
    val sinks = currentCallSites.map{ case (ctorname, args) =>
      val matchedSinks: Seq[Sink] = trail.sinks.collect{
        case sink if sameSinkConstructor(sink, ctorname) => sink
      }
      (matchedSinks.headOption, ctorname, args, matchedSinks.nonEmpty)
    }
    val nonEmptySinks = sinks.filter{ case (_, _, _, item) => item}
    for {
      (optSink, ctorname, args, _) <- nonEmptySinks
      sink <- optSink
    } yield CallSite(ctorname, args, sink)
  }

  private[this] def treeHasImport(importElements: Seq[String]): Boolean = {
    tree.collect{
      case q"import ..$importersnel" if importersnel.exists{ case importer =>
        importElements.contains(importer.toString)} => true
    }.exists(identity)
  }

  private[this] def isHardCodedString(arg: Term) = {
    arg.collect{
      case q"${lit: Lit}" if isCharOrStringLiteral(lit.value) => true
    }.nonEmpty
  }

  private[this] def isHardCodedArray(arg: Tree) = {
    arg.collect{
      case q"${lit: Lit}.toCharArray" if isCharOrStringLiteral(lit.value) => true
      case init"${ctorname: Type}(..${args: Seq[Term]})"
        if ctorname.toString == "Array" && args.forall(isCharOrStringLiteral) => true
    }.nonEmpty
  }

  private[this] def isCharOrStringLiteral(value: Any): Boolean = {
    value match {
      case t: String => true
      case t: Char => true
      case _ => false
    }
  }

  private[this] def isHardCodedVariable(arg: Term, callSite: CallSite) = {
    arg.collect{
      case q"${name: Term.Name}" if isSuspiciousName(name, callSite) => true
    }.headOption.fold(false)(_ => true)
  }

  private[this] def isValueHardcoded(arg: Term) = {
    isHardCodedString(arg) || isHardCodedArray(arg)
  }

  private[this] def hardCodedValue(callSite: CallSite): Option[Term] = {
    val position = callSite.sink.position
    val callSiteArg = callSite.args.lift(position)
    callSiteArg match {
      case Some(arg) if isValueHardcoded(arg) || isHardCodedVariable(arg, callSite) => Some(arg)
      // Can't find the particular argument.
      case _ => Option.empty
    }
  }

  def findHardCodedPasswords(): Seq[Term] = {
    val foundCallSites = matchedCallSites()
    foundCallSites.flatMap{ case callSite => hardCodedValue(callSite)}
  }

  def trailHasHints(): Boolean = {
    trail.sinks.exists{ case sink => treeHasImport(sink.hints) }
  }
}

case object Custom_Scala_HardCodedPassword extends Pattern{

  override def apply(tree: Tree) = {
    val trails = Seq(
      new SprayTrail(tree),
      new JavaTrail(tree),
      new FinagleTrail(tree)
    )
    val finders = trails.map { trail => new Finder(tree, trail)}
      .filter(_.trailHasHints())
    val places = finders.flatMap(_.findHardCodedPasswords())
    places.map { case item => Result(message(item), item)}
  }

  private[this] def message(tree: Tree) = Message("Hard coded password")

}