# Quote this to prevent word splitting

### Problematic code:

    ls -l $(getfilename)

### Correct code:

    # getfilename outputs 1 file
    ls -l "$(getfilename)"

    # getfilename outputs multiple files, linefeed separated
    getfilename | while IFS='' read -r line
    do
      ls -l "$line"
    done

### Rationale:

When command expansions are unquoted, word splitting and globbing will occur. This often manifests itself by breaking when filenames contain spaces.

Trying to fix it by adding quotes or escapes to the data will not work. Instead, quote the command substitution itself.

If the command substitution outputs multiple pieces of data, use a loop instead.

### Exceptions

In rare cases you actually want word splitting, such as in

    gcc $(pkg-config --libs openssl) client.c

This is because `pkg-config` outputs `-lssl -lcrypto`, which you want to break up by spaces into `-lssl` and `-lcrypto`. An alternative is to put the variables to an array and expand it:

    # For bash
    mapfile -t args < <(pkg-config --libs openssl)
    
    # For ksh
    pkg-config --libs openssl | while IFS="" read -r line; do array+=("$line"); done
    
    # expand args
    gcc "${args[@]}" client.c

The power of using an array becomes evident when you want to combine, for example, the command result with user-provided arguments:

    compile () {
        args=( $(pkg-config --libs openssl) "${@}" )
        gcc "${args[@]}" client.c
    }
    compile -DDEBUG
    + gcc -lssl -lcrypto -DDEBUG client.c
[Source](https://github.com/koalaman/shellcheck/wiki/SC2046)

