## Bash doesn't support variables in brace range expansions.

### Problematic code:

```bash
for i in {1..$n}
do
  echo "$i"
done
```

### Correct code:

```bash
for ((i=0; i<n; i++))
do
  echo "$i"
done
```

### Rationale:

In Bash, brace expansion happens before variable expansion. This means that brace expansion will not account for variables.

For integers, use an arithmetic for loop instead. For zero-padded numbers or letters, use of eval may be warranted:

```bash
from="a" to="m"
for c in $(eval "echo {$from..$to}"); do echo "$c"; done
```

or more carefully (if `from`/`to` could be user input, or if the brace expansion could have spaces):

```bash
from="a" to="m"
while IFS= read -d '' -r c
do
  echo "Read $c"
done <  <(eval "printf '%s\0' $(printf "{%q..%q}.jpg" "$from" "$to")")
```

### Exceptions

None (if you're writing for e.g. zsh, make sure the shebang indicates this so shellcheck won't warn)

### Related Resources:

* [StackOverflow: Variables in bash seq replacement ({1..10})](https://stackoverflow.com/questions/169511/how-do-i-iterate-over-a-range-of-numbers-defined-by-variables-in-bash)
[Source](https://github.com/koalaman/shellcheck/wiki/SC2051)

