Using the equality `==` and inequality `!=` operators to compare two objects generally works. The operators can be
overloaded, and therefore the comparison can resolve to the appropriate method. However, when the operators are used on interface instances, then
`==` resolves to reference equality, which may result in unexpected behavior if implementing classes override `Equals`.
Similarly, when a class overrides `Equals`, but instances are compared with non-overloaded `==`, there is a high chance that
value comparison was meant instead of the reference one.

## Noncompliant Code Example

<pre>
public interface IMyInterface
{
}

public class MyClass : IMyInterface
{
    public override bool Equals(object obj)
    {
        //...
    }
}

public class Program
{
    public static void Method(IMyInterface instance1, IMyInterface instance2)
    {
        if (instance1 == instance2) // Noncompliant, will do reference equality check, but was that intended? MyClass overrides Equals.
        {
            Console.WriteLine("Equal");
        }
    }
}
</pre>

## Compliant Solution

<pre>
public interface IMyInterface
{
}

public class MyClass : IMyInterface
{
    public override bool Equals(object obj)
    {
        //...
    }
}

public class Program
{
    public static void Method(IMyInterface instance1, IMyInterface instance2)
    {
        if (object.Equals(instance1, instance2)) // object.Equals checks for null and then calls the instance based Equals, so MyClass.Equals
        {
            Console.WriteLine("Equal");
        }
    }
}
</pre>

## Exceptions

The rule does not report on comparisons of `System.Type` instances and on comparisons inside `Equals` overrides.

## See

*   [MITRE, CWE-595](http://cwe.mitre.org/data/definitions/595.html) - Comparison of Object References Instead of Object Contents
*   [MITRE, CWE-597](http://cwe.mitre.org/data/definitions/597.html) - Use of Wrong Operator in String Comparison
*   [CERT, EXP03-J.](https://www.securecoding.cert.org/confluence/x/wwD1AQ) - Do not use the equality operators when comparing values of
      boxed primitives
*   [CERT, EXP50-J.](https://www.securecoding.cert.org/confluence/x/8AEqAQ) - Do not confuse abstract object equality with reference
      equality