## Avoid hardcoding paths to temp files

### Problematic code:

```python
import os

with open('/tmp/foo', 'w') as f:
    f.write('hello world')

os.remove('/tmp/foo')
```

### Correct code:

```python
import tempfile

with tempfile.TemporaryFile() as f:
    f.write('hello world')
```

### Rationale:

Creating a predictably-named temporary file allows a malicious user to replace the file or its contents with their own data. Additionally, different systems may configure temp paths differently.

Python provides several tools in the `tempfile` module. `TemporaryFile` creates a securely-named file and, when used in a `with` statement, cleans up the file afterward.

If you need to retain the temp file and (for example) expose it for another program, use `tempfile.mkstemp` or use `tempfile.mkdtemp` to create a temp directory.

[Reference](https://docs.openstack.org/bandit/latest/plugins/b108_hardcoded_tmp_directory.html)

