package codacy.patterns

import codacy.base.Pattern

import scala.meta._

case object Custom_Scala_PotentialFileNameInjection extends Pattern {

  override def apply(tree: Tree): Seq[Result] = {
    if (hasReaderHints(tree)) {
      val unsafeArguments: List[Term] = findUnsafeArguments(tree)
      findUnsafeReaderParameters(tree, unsafeArguments)
    } else Seq.empty
  }

  private[this] def findUnsafeArguments(tree: Tree) = {
    val nodes = tree.collect {
      case q"Source.fromFile(...$exprss)" => exprss.headOption.flatMap(_.headOption)
      case init"FileReader(..${args: Seq[Term]})" => args.headOption
    }.flatten
    nodes.filter { case arg =>
      arg match {
        case q"${arg: Term.Name}" => true
        case _ => false
      }
    }
  }

  private[this] def isStringParam(param: Term.Param): Boolean = {
    param.collect{
      case param"..$mods $paramname: ${atpeopt: Option[Type]} = $exprop" if atpeopt.getOrElse("").toString == "String" =>
        true
    }.nonEmpty
  }

  private[this] def scopeParams(tree: Tree): List[Name] = {
    tree.collect {
      case t@q"..$mods def $name[..$tparams](..${paramss: Seq[Term.Param]}): $tpeopt = ${expr: Tree}" if t == tree =>
        paramss.filter(isStringParam).map(_.name)
      case t@q"..$mods class $name [..$tparams] (..${paramss: Seq[Term.Param]}) extends ..$supers {$body}" if t == tree =>
        paramss.filter(isStringParam).map(_.name)
    }.flatten
  }

  private[this] def determineScope(node: Tree): Option[Tree] = {
    node match {
      case t@q"..$mods def $name[..$tparams](...$paramss): $tpeopt = $expr" => Some(t)
      case t@q"..$mods class $name (...$params) extends $template" => Some(t)
      case source"..$stats" => Option.empty
      case item => item.parent.flatMap(determineScope)
    }
  }

  private[this] def findUnsafeReaderParameters(tree: Tree, args: List[Term]) = {
    args.filter{ case argument =>
        val argumentScope = argument.parent.fold(Option.empty[Tree])(determineScope)
        val params = argumentScope.fold(Seq.empty[Name])(scopeParams)
        params.map(_.toString).contains(argument.toString)
    }.map { case arg => Result(message(arg), arg)}
  }

  private[this] def hasReaderHints(tree: Tree): Boolean = {
    tree.collect{
      case importer"java.io._" => true
      case importer"scala.io" => true
    }.exists(identity)
  }

  private[this] def message(tree: Tree) = Message("Potential filename injection.")
}
