package codacy.patterns

import codacy.base.{Pattern, Result}

import scala.meta._

case class Custom_Scala_InsufficientKeySizeRsaDetector(conf:Custom_Scala_InsufficientKeySizeRsaDetector.Configuration) extends AnyVal with Pattern{
  override def apply(tree: Tree): Iterable[Result] = {
    tree.collect{
      case t@q"""${selector: Term}.getInstance("RSA")""" if isEvilSelector(selector, selector.parent.exists(hasEvilImport(_,t.pos))) =>
        variableNamesAndScopeFor((t:Tree)).toSeq.flatMap{ case (scope:Tree,names:Seq[Name]) =>
          inits(names,scope,conf.minSize)
        }
    }.flatten.map{ case tree => Result(message(tree),tree) }
  }

  private[this] def message(tree: Tree) = Message("RSA usage with short key")

  private[this] def isEvilSelect(tree:Tree) = tree match{
    case q"java.security" => true
    case _ => false
  }

  private[this] def isEvilName(name:Name) = {
    name.value == "KeyPairGenerator"
  }

  // do we have the import? if yes invocations of KeyPairGenerator are problematic
  // if no we just check for the fully qualified java.security.KeyPairGenerator
  private[this] def hasEvilImport(tree:Tree,pos:Position):Boolean = {

    val foundInThisScope = tree.collect{
      case t@importer"$select.{..${importeesnel:Seq[Importee]} }" if t.pos.isBefore(pos) && isEvilSelect(select) =>
        importeesnel
    }.flatten.exists{
      case importee"_" => true

      case importee"${name:Name}" => isEvilName(name)

      case _ => false
    }

    foundInThisScope || tree.parent.exists(hasEvilImport(_,pos))
  }

  private[this] def isEvilSelector(term:Term,hasImport: => Boolean ): Boolean = {
    term match{
      case q"$sel.${name:Term.Name}" =>
        isEvilSelect(sel) && isEvilSelector(name,true)

      case q"${name:Name}" =>
        isEvilName(name) && hasImport

      case _ => false
    }
  }

  /* give me the names of variables/methods and the scope in which they are valid */
  private[this] def variableNamesAndScopeFor(tree:Tree):Option[(Tree,Seq[Name])] = {
    tree.parent.flatMap{
      case block@q"{ ..${stats:Seq[Stat]} }" if stats.lastOption.exists(_ == tree) =>
        variableNamesAndScopeFor(block)

      case t@q"""..$_ val ..${patsnel: Seq[Pat]}: $_ = $_""" =>
        t.parent.map{ case scope => (scope,namesForPatterns(patsnel)) }

      case t@q"""..$_ var ..${patsnel: Seq[Pat]}: $_ = $_""" =>
        t.parent.map{ case scope => (scope,namesForPatterns(patsnel)) }

      case t@q"..$mods def ${name: Name}[..$tparams](...$paramss): $tpeopt = $expr" =>
        t.parent.map{ case scope => (scope,Seq(name)) }

      case _ =>
        Option.empty
    }
  }

  private[this] def namesForPatterns(patsnel:Seq[Pat]) = {
    patsnel.collect{ case q"${ name: Pat.Var }" => name.name }
  }

  private[this] def inits(names:Seq[Name],block:Tree, minSize:Int):Seq[Tree] = {
    block.collect{
      case t@q"${term: Term}.initialize(${lit: Lit})" if isTooSmall(lit,minSize) && isEvilInit(term,names) =>
        t
    }
  }

  def isEvilInit(term:Term, names:Seq[Name]):Boolean = {
    lazy val allNames = names.map(_.value)

    term match{
      case q"${name:Name}" =>
        allNames.contains(name.value)
      case q"${name : Term.Name }(..$_)" =>
        allNames.contains(name.value)
      case q"${name : Term.Name }[$_](..$_)" =>
        allNames.contains(name.value)
    }
  }

  private[this] def isTooSmall(sizeParam:Lit, minSize:Int):Boolean = {
    sizeParam.value match{
      case l:Int => l < minSize
      case _ => false
    }
  }
}

case object Custom_Scala_InsufficientKeySizeRsaDetector{
  case class Configuration(minSize:Int=1024)
}
