## Remove surrounding $() to avoid executing output.

### Problematic code:

```sh
if $(which epstopdf)
then
  echo "Found epstopdf"
fi
```

### Correct code:

```sh
if which epstopdf
then
  echo "Found epstopdf"
fi
```

### Rationale:

ShellCheck has detected that you have a command that just consists of a command substitution.

This is typically done in order to try to get the shell to execute a command, because `$(..)` does indeed execute commands. However, it's also replaced by the output of that command.

When you run `echo "The date is $(date +%F)"`, bash evalutes the `$(..)`. The command then becomes `echo "The date is 2015-04-29"`, which writes out the string `The date is 2015-04-29`

The problem is when you use `$(date +%F)` alone as a command. Bash evaluates the `$(..)`, and the command then becomes `2015-04-29`. There is no command called `2015-04-29`, so you get `bash: 2015-04-29: command not found`.

Sometimes this results in this confounding `command not found` messages. Other times you get even stranger issues, like the example problematic code which always evaluates to false.

The solution is simply to remove the surrounding `$()`. This will execute the command instead of the command's output.

### Exceptions:

If you really want to execute the output of a command rather than the command itself, you can ignore this message or assign the output to a new variable first:
```sh
readonly command_to_execute="$(print_the_command)"
$command_to_execute
```

### Related resources:

* StackOverflow: [Bash Function -> Command not found](https://stackoverflow.com/questions/18611842/bash-function-command-not-found)
[Source](https://github.com/koalaman/shellcheck/wiki/SC2091)

