
The `IDisposable` interface is a mechanism to release unmanaged resources, if not implemented correctly this could result in resource<br>leaks or more severe bugs.

This rule raises an issue when the recommended dispose pattern, as defined by Microsoft, is not adhered to. See the **Compliant<br>Solution** section for examples.

Satisfying the rule's conditions will enable potential derived classes to correctly dispose the members of your class:

- `sealed` classes are not checked.
- If a base class implements `IDisposable` your class should not have `IDisposable` in the list of its interfaces. In such<br>  cases it is recommended to override the base class's `protected virtual void Dispose(bool)` method or its equivalent.
- The class should not implement `IDisposable` explicitly, e.g. the `Dispose()` method should be public.
- The class should contain `protected virtual void Dispose(bool)` method. This method allows the derived classes to correctly dispose<br>  the resources of this class.
- The content of the `Dispose()` method should be invocation of `Dispose(true)` followed by<br>  `GC.SuppressFinalize(this)`
- If the class has a finalizer, i.e. a destructor, the only code in its body should be a single invocation of `Dispose(false)`.
- If the class inherits from a class that implements `IDisposable` it must call the `Dispose`, or<br>  `Dispose(bool)` method of the base class from within its own implementation of `Dispose` or `Dispose(bool)`,<br>  respectively. This ensures that all resources from the base class are properly released.


## Noncompliant Code Example


    public class Foo1 : IDisposable // Noncompliant - provide protected overridable implementation of Dispose(bool) on Foo or mark the type as sealed.
    {
        public void Dispose() // Noncompliant - should contain only a call to Dispose(true) and then GC.SuppressFinalize(this)
        {
            // Cleanup
        }
    }
    
    public class Foo2 : IDisposable
    {
        void IDisposable.Dispose() // Noncompliant - Dispose() should be public
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }
    
        public virtual void Dispose() // Noncompliant - Dispose() should be sealed
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }
    }
    
    public class Foo3 : IDisposable
    {
        public void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }
    
        protected virtual void Dispose(bool disposing)
        {
            // Cleanup
        }
    
        ~Foo3() // Noncompliant - Modify Foo.~Foo() so that it calls Dispose(false) and then returns.
        {
            // Cleanup
        }
    }{code}


## Compliant Solution


    // Sealed class
    public sealed class Foo1 : IDisposable
    {
        public void Dispose()
        {
            // Cleanup
        }
    }
    
    // Simple implementation
    public class Foo2 : IDisposable
    {
        public void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }
    
        protected virtual void Dispose(bool disposing)
        {
            // Cleanup
        }
    }
    
    // Implementation with a finalizer
    public class Foo3 : IDisposable
    {
        public void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }
    
        protected virtual void Dispose(bool disposing)
        {
            // Cleanup
        }
    
        ~Foo3()
        {
            Dispose(false);
        }
    }
    
    // Base disposable class
    public class Foo4 : DisposableBase
    {
        protected override void Dispose(bool disposing)
        {
            // Cleanup
            // Do not forget to call base
            base.Dispose(disposing);
        }
    }


## See

Refer to

- [MSDN](https://msdn.microsoft.com/en-us/library/498928w2.aspx) for complete documentation on the dispose pattern.
- [Stephen Cleary](http://blog.stephencleary.com/2009/08/how-to-implement-idisposable-and.html) for excellent Q&A about<br>  IDisposable
- [Pragma Geek](http://pragmateek.com/c-scope-your-global-state-changes-with-idisposable-and-the-using-statement/) for additional<br>  usages of IDisposable, beyond releasing resources.
- [IDisposable documentation](https://docs.microsoft.com/en-us/dotnet/api/system.idisposable?view=netframework-4.7)

