# To read lines rather than words, pipe/redirect to a 'while read' loop.

### Problematic code:

```sh
for line in $(cat file | grep -v '^ *#')
do
  echo "Line: $line"
done
```

### Correct code:

```sh
grep -v '^ *#' < file | while IFS= read -r line
do
  echo "Line: $line"
done
```

or without a subshell (bash, zsh, ksh):

```sh
while IFS= read -r line
do
  echo "Line: $line"
done < <(grep -v '^ *#' < file)
```

or without a subshell, with a pipe (more portable, but write a file on the filesystem):

```sh
mkfifo mypipe
grep -v '^ *#' < file > mypipe &
while IFS= read -r line
do
  echo "Line: $line"
done < mypipe
rm mypipe
```

### Rationale:

For loops by default (subject to `$IFS`) read word by word. Additionally, glob expansion will occur.

Given this text file:

```sh
foo *
bar
```

The for loop will print:

```sh
Line: foo
Line: aardwark.jpg
Line: bullfrog.jpg
...
```

The while loop will print:

```sh
Line: foo *
Line: bar
```


### Exceptions

If you do want to read word by word, you can set `$IFS` appropriately and disable globbing with `set -f`, and then [[ignore]] this warning. Alternatively, you can pipe through `tr ' ' '\n'` to turn words into lines, and then use `while read`. In Bash/Ksh, you can also use a `while read -a` loop to get an array of words per line.


[Source](https://github.com/koalaman/shellcheck/wiki/SC2013)

