## Use cd ... || exit in case cd fails.

### Problematic code:

```sh
cd generated_files
rm -r *.c
```
```sh
func(){
  cd foo
  do_something
}
```

### Correct code:

```sh
cd generated_files || exit
rm -r *.c
```
```sh
# For functions, you may want to use return:
func(){
  cd foo || return
  do_something
}
```

### Rationale:

`cd` can fail for a variety of reasons: misspelled paths, missing directories, missing permissions, broken symlinks and more.

If/when it does, the script will keep going and do all its operations in the wrong directory. This can be messy, especially if the operations involve creating or deleting a lot of files.

To avoid this, make sure you handle the cases when `cd` fails. Ways to do this include

* `cd foo || exit` as suggested to abort immediately, reusing exit code from failed `cd` command
* `cd foo || { echo "Failure"; exit 1; }` abort with custom message
* `if cd foo; then echo "Ok"; else echo "Fail"; fi` for custom handling
* `<(cd foo && cmd)` as an alternative to `<(cd foo || exit; cmd)` in `<(..)`, `$(..)` or `( )`

### Exceptions:

ShellCheck does not give this warning when `cd` is on the left of a `||` or `&&`, or the condition of a `if`, `while` or `until` loop. Having a `set -e` command anywhere in the script will disable this message, even though it won't necessarily prevent the issue.

If you are accounting for `cd` failures in a way shellcheck doesn't realize, you can disable this message with a [[directive]].

[Source](https://github.com/koalaman/shellcheck/wiki/SC2164)

