
Formatting strings used as SQL queries is security-sensitive. It has led in the past to the following vulnerabilities:

- [CVE-2018-9019](http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2018-9019)
- [CVE-2018-7318](http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2018-7318)
- [CVE-2017-5611](http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2017-5611)


SQL queries often need to use a hardcoded SQL string with a dynamic parameter coming from a user request. Formatting a string to add those<br>parameters to the request is a bad practice as it can result in an [SQL injection](https://www.owasp.org/index.php/SQL_Injection). The safe<br>way to add parameters to a SQL query is to use SQL binding mechanisms.

This rule flags the execution of SQL queries which are built using formatting of strings, even if there is no injection. This rule does not detect<br>SQL injections. The goal is to guide security code reviews and to prevent a common bad practice.

The following specific method signatures are tested:

- `System.Data.SqlClient.SqlCommand.SqlCommand(string, ...)`
- `System.Data.SqlClient.SqlDataAdapter.SqlDataAdapter(string, ...)`
- `System.Data.Odbc.OdbcCommand.OdbcCommand(string, ...)`
- `System.Data.Odbc.OdbcDataAdapter.OdbcDataAdapter(string, ...)`
- `System.Data.SqlServerCe.SqlCeCommand.SqlCeCommand(string, ...)`
- `System.Data.SqlServerCe.SqlCeDataAdapter.SqlCeDataAdapter(string, ...)`
- `System.Data.OracleClient.OracleCommand.OracleCommand(string, ...)`
- `System.Data.OracleClient.OracleDataAdapter.OracleDataAdapter(string, ...)`
- `Microsoft.EntityFrameworkCore.RelationalDatabaseFacadeExtensions.ExecuteSqlCommand(...)`
- `Microsoft.EntityFrameworkCore.RelationalDatabaseFacadeExtensions.ExecuteSqlCommandAsync(...)`
- `Microsoft.EntityFrameworkCore.RelationalQueryableExtensions.FromSql<TEntity>(System.Linq.IQueryable<TEntity>,
  System.FormattableString)`
- `System.Data.SqlClient.SqlCommand.CommandText.set`
- `System.Data.Odbc.OdbcCommand.CommandText.set`
- `System.Data.SqlServerCe.SqlCeCommand.CommandText.set`
- `System.Data.OracleClient.OracleCommand.CommandText.set`


The following formatting methods will raise an issue:

- `String.Concat`
- `String.Format`
- `FormattableString`, with some exceptions (see the Exceptions section)


## Ask Yourself Whether

- the SQL query is built using string formatting technics, such as concatenating variables.
- some of the values are coming from an untrusted source and are not sanitized.


You may be at risk if you answered yes to this question.

## Recommended Secure Coding Practices

- Avoid building queries manually using formatting technics. If you do it anyway, do not include user input in this building process.
- Use [parameterized queries, prepared statements, or stored<br>  procedures](https://www.owasp.org/index.php/Query_Parameterization_Cheat_Sheet) whenever possible.
- You may also use ORM frameworks such as Hibernate which, if used correctly, reduce injection risks.
- Avoid executing SQL queries containing unsafe input in stored procedures or functions.
- [Sanitize](https://www.owasp.org/index.php/Input_Validation_Cheat_Sheet) every unsafe input.


You can also reduce the impact of an attack by using a database account with low privileges.

## Sensitive Code Example


    public void Foo(DbContext context, string query, string param)
    {
        string sensitiveQuery = string.Concat(query, param);
        context.Database.ExecuteSqlCommand(sensitiveQuery); // Sensitive
        context.Query&lt;User&gt;().FromSql(sensitiveQuery); // Sensitive
    
        context.Database.ExecuteSqlCommand($"SELECT * FROM mytable WHERE mycol={value}", param); // Sensitive, the FormattableString is evaluated and converted to RawSqlString
        string query = $"SELECT * FROM mytable WHERE mycol={param}";
        context.Database.ExecuteSqlCommand(query); // Sensitive, the FormattableString has already been evaluated, it won't be converted to a parametrized query.
    }
    
    public void Bar(SqlConnection connection, string param)
    {
        SqlCommand command;
        string sensitiveQuery = string.Format("INSERT INTO Users (name) VALUES (\"{0}\")", param);
        command = new SqlCommand(sensitiveQuery); // Sensitive
    
        command.CommandText = sensitiveQuery; // Sensitive
    
        SqlDataAdapter adapter;
        adapter = new SqlDataAdapter(sensitiveQuery, connection); // Sensitive
    }


## Exceptions

No issue will be raised in the following cases:

- When the SQL query is a FormattableString provided **directly** to `ExecuteSqlCommand`,<br>  `ExecuteSqlCommandAsync` or `FromSql`. In this case the FormattableString is automatically converted to a parametrized query.<br>  This feature was added in Entity Framework Core 2.0. See [Entity Framework's<br>  documentation](https://docs.microsoft.com/en-us/ef/core/what-is-new/ef-core-2.0#string-interpolation-in-fromsql-and-executesqlcommand) for more information.



    public void Foo(DbContext context, string value)
    {
        context.Database.ExecuteSqlCommand("SELECT * FROM mytable"); // No issue raised. The query is hard-coded. Thus no injection is possible.
    
        context.Database.ExecuteSqlCommand($"SELECT * FROM mytable WHERE mycol={value}"); // No issue raised. The FormattableString is transformed into a parametrized query.
    }


## See

- [OWASP Top 10 2017 Category A1](https://www.owasp.org/index.php/Top_10-2017_A1-Injection) - Injection
- [MITRE, CWE-89](http://cwe.mitre.org/data/definitions/89) - Improper Neutralization of Special Elements used in an SQL Command
- [MITRE, CWE-564](http://cwe.mitre.org/data/definitions/564.html) - SQL Injection: Hibernate
- [MITRE, CWE-20](http://cwe.mitre.org/data/definitions/20.html) - Improper Input Validation
- [MITRE, CWE-943](http://cwe.mitre.org/data/definitions/943.html) - Improper Neutralization of Special Elements in Data Query Logic<br>
- [CERT, IDS00-J.](https://www.securecoding.cert.org/confluence/x/PgIRAg) - Prevent SQL injection
- [SANS Top 25](https://www.sans.org/top25-software-errors/#cat1) - Insecure Interaction Between Components
- Derived from FindSecBugs rules [Potential SQL/JPQL Injection<br>  (JPA)](http://h3xstream.github.io/find-sec-bugs/bugs.htm#SQL_INJECTION_JPA), [Potential SQL/JDOQL Injection (JDO)](http://h3xstream.github.io/find-sec-bugs/bugs.htm#SQL_INJECTION_JDO), [Potential SQL/HQL Injection (Hibernate)](http://h3xstream.github.io/find-sec-bugs/bugs.htm#SQL_INJECTION_HIBERNATE)

