
Encrypting data is security-sensitive. It has led in the past to the following vulnerabilities:

- [CVE-2017-7902](http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2017-7902)
- [CVE-2006-1378](http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2006-1378)
- [CVE-2003-1376](http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2003-1376)


Proper encryption requires both the encryption algorithm and the key to be strong. Obviously the private key needs to remain secret and be renewed<br>regularly. However these are not the only means to defeat or weaken an encryption.



This rule flags function calls that initiate encryption/decryption.

## Ask Yourself Whether

- the private key might not be random, strong enough or the same key is reused for a long long time.
- the private key might be compromised. It can happen when it is stored in an unsafe place or when it was transferred in an unsafe manner.
- the key exchange is made without properly authenticating the receiver.
- the encryption algorithm is not strong enough for the level of protection required. Note that encryption algorithms strength decreases as time<br>  passes.
- the chosen encryption library is deemed unsafe.
- a nonce is used, and the same value is reused multiple times, or the nonce is not random.
- the RSA algorithm is used, and it does not incorporate an Optimal Asymmetric Encryption Padding (OAEP), which might weaken the encryption.<br>
- the CBC (Cypher Block Chaining) algorithm is used for encryption, and it's IV (Initialization Vector) is not generated using a secure random<br>  algorithm, or it is reused.
- the Advanced Encryption Standard (AES) encryption algorithm is used with an unsecure mode. See the recommended practices for more information.<br>


You are at risk if you answered yes to any of those questions.

## Recommended Secure Coding Practices

- Generate encryption keys using secure random algorithms.
- When generating cryptographic keys (or key pairs), it is important to use a key length that provides enough entropy against brute-force<br>  attacks. For the Blowfish algorithm the key should be at least 128 bits long, while for the RSA algorithm it should be at least 2048 bits long.<br>
- Regenerate the keys regularly.
- Always store the keys in a safe location and transfer them only over safe channels.
- If there is an exchange of cryptographic keys, check first the identity of the receiver.
- Only use strong encryption algorithms. Check regularly that the algorithm is still deemed secure. It is also imperative that they are<br>  implemented correctly. Use only encryption libraries which are deemed secure. Do not define your own encryption algorithms as they will most<br>  probably have flaws.
- When a nonce is used, generate it randomly every time.
- When using the RSA algorithm, incorporate an Optimal Asymmetric Encryption Padding (OAEP).
- When CBC is used for encryption, the IV must be random and unpredictable. Otherwise it exposes the encrypted value to crypto-analysis attacks<br>  like "Chosen-Plaintext Attacks". Thus a secure random algorithm should be used. An IV value should be associated to one and only one encryption<br>  cycle, because the IV's purpose is to ensure that the same plaintext encrypted twice will yield two different ciphertexts.
- The Advanced Encryption Standard (AES) encryption algorithm can be used with various modes. Galois/Counter Mode (GCM) with no padding should be<br>  preferred to the following combinations which are not secured:
    - Electronic Codebook (ECB) mode: Under a given key, any given plaintext block always gets encrypted to the same ciphertext block. Thus, it<br>      does not hide data patterns well. In some senses, it doesn't provide serious message confidentiality, and it is not recommended for use in<br>      cryptographic protocols at all.
    - Cipher Block Chaining (CBC) with PKCS#5 padding (or PKCS#7) is susceptible to padding oracle attacks.


## Sensitive Code Example


    using System;
    using System.Security.Cryptography;
    
    namespace MyNamespace
    {
        public class MyClass
        {
            public void Main()
            {
                Byte[] data = {1,1,1};
    
                RSA myRSA = RSA.Create();
                RSAEncryptionPadding padding = RSAEncryptionPadding.CreateOaep(HashAlgorithmName.SHA1);
                // Review all base RSA class' Encrypt/Decrypt calls
                myRSA.Encrypt(data, padding); // Sensitive
                myRSA.EncryptValue(data); // Sensitive
                myRSA.Decrypt(data, padding); // Sensitive
                myRSA.DecryptValue(data); // Sensitive
    
                RSACryptoServiceProvider myRSAC = new RSACryptoServiceProvider();
                // Review the use of any TryEncrypt/TryDecrypt and specific Encrypt/Decrypt of RSA subclasses.
                myRSAC.Encrypt(data, false); // Sensitive
                myRSAC.Decrypt(data, false); // Sensitive
                int written;
                myRSAC.TryEncrypt(data, Span&lt;byte&gt;.Empty, padding, out written); // Sensitive
                myRSAC.TryDecrypt(data, Span&lt;byte&gt;.Empty, padding, out written); // Sensitive
    
                byte[] rgbKey = {1,2,3};
                byte[] rgbIV = {4,5,6};
                SymmetricAlgorithm rijn = SymmetricAlgorithm.Create();
                // Review the creation of Encryptors from any SymmetricAlgorithm instance.
                rijn.CreateEncryptor(); // Sensitive
                rijn.CreateEncryptor(rgbKey, rgbIV); // Sensitive
                rijn.CreateDecryptor(); // Sensitive
                rijn.CreateDecryptor(rgbKey, rgbIV); // Sensitive
            }
    
            public class MyCrypto : System.Security.Cryptography.AsymmetricAlgorithm // Sensitive
            {
                // ...
            }
    
            public class MyCrypto2 : System.Security.Cryptography.SymmetricAlgorithm // Sensitive
            {
                // ...
            }
        }
    }


## See

- [OWASP Top 10 2017 Category A3](https://www.owasp.org/index.php/Top_10-2017_A3-Sensitive_Data_Exposure) - Sensitive Data Exposure<br>
- [OWASP Top 10 2017 Category A6](https://www.owasp.org/index.php/Top_10-2017_A6-Security_Misconfiguration) - Security<br>  Misconfiguration
- [MITRE, CWE-321](http://cwe.mitre.org/data/definitions/321.html) - Use of Hard-coded Cryptographic Key
- [MITRE, CWE-322](http://cwe.mitre.org/data/definitions/322.html) - Key Exchange without Entity Authentication
- [MITRE, CWE-323](http://cwe.mitre.org/data/definitions/323.html) - Reusing a Nonce, Key Pair in Encryption
- [MITRE, CWE-324](http://cwe.mitre.org/data/definitions/324.html) - Use of a Key Past its Expiration Date
- [MITRE, CWE-325](http://cwe.mitre.org/data/definitions/325.html) - Missing Required Cryptographic Step
- [MITRE, CWE-326](http://cwe.mitre.org/data/definitions/326.html) - Inadequate Encryption Strength
- [MITRE, CWE-327](http://cwe.mitre.org/data/definitions/327.html) - Use of a Broken or Risky Cryptographic Algorithm
- [MITRE, CWE-522](http://cwe.mitre.org/data/definitions/522.html) - Insufficiently Protected Credentials
- [SANS Top 25](https://www.sans.org/top25-software-errors/#cat3) - Porous Defenses

